#' @title Simulate survival endpoints data where censoring depends on missing covariates.
#' @description We are assuming three covariates, two continuous and one binary,
#' where we assume the first continous and the binary covariates have missing values, and the second continuous covariates is fully observed.
#' @param n Sample Size.
#' @param beta Regression Coefficients for the event process (length of 3).
#' @param phi Regression Coefficients for the censoring process (length of 3).
#' @param gamma Regression Coefficients for the missingness (length of 3).
#' @param seed A random seed for data generation.
#'
#' @return A data frame with missing values. For the \code{delta} column, 0 = censored, and 1 = event.
#' @importFrom stats rnorm rbinom runif
#'
#' @examples
#' # Simulate a data set with approximately 50% censoring
#' # and 50% of the subjects with at least one missing covariates.
#' generate.surv(n = 100, beta = c(1,1,-1), phi= c(1,-1,-0.5), gamma = c(3,2,-1), seed = 112358)
#' @export
generate.surv <- function(n, beta, phi, gamma, seed){
  set.seed(seed)
  x1 <- rnorm(n, mean = 0, sd = 1)
  x2 <- rbinom(n, 1, prob = 0.5)
  x3 <- rnorm(n, mean = 2, sd = 1)
  X <- matrix(c(x1,x2,x3), ncol = 3)

  u <-  runif(n, 0, 1)
  t <-  -log(u)/exp(X%*%beta)

  v <- runif(n, 0, 1)
  c <- -log(v)/exp(X%*%phi)
  time = pmin(t, c)
  delta = ifelse(t <= c, 1, 0)
  df <- data.frame(time, delta, X, t)

  Z <- matrix(c(rep(1, times = n), df$X3, df$t), ncol = 3)
  mu <- exp(Z%*%gamma)/(1 + exp(Z%*%gamma))
  m <- rbinom(n, 1, mu)
  type <- sample(1:5, n, replace = TRUE)
  df$X1[m == 1 & type == 1] <- NA
  df$X1[m == 1 & type == 2] <- NA

  df$X2[m == 1 & type == 3] <- NA

  return(df[,-6])
}


