#' SeasEpi for Real Data
#'
#' This function applies the spatiotemporal individual-level model of seasonal infectious disease transmission within the Susceptible-Exposed-Infectious-Recovered-Susceptible (SEIRS) framework, to real data. It employs a likelihood based Monte Carlo Expectation Conditional Maximization (MCECM) algorithm for parameter estimation and AIC calculation. This function requires two dataframes, named `data` and `adjacency_matrix`, along with the necessary parameters. Detailed information on the structure of these two datasets is provided in the package.
#' @param data Dataset. The dataset should exactly match the `data` file in the data folder, including all the columns with the same names.
#' @param adjacency_matrix Adjacency matrix representing the regions in the study area (0 if no connection between regions)
#' @param DimCovInf Dimensions of the individual infectivity covariate
#' @param DimCovSus Dimensions of the area-level susceptibility to initial infection covariate
#' @param tau0 Initial value for spatial precision
#' @param lambda0 Initial value for spatial dependence
#' @param alphaS0 Initial value for the susceptibility intercept
#' @param delta0 Initial value for the spatial decay parameter
#' @param zeta10 Initial value for the amplitude of the seasonal oscillation parameter (sin part)
#' @param zeta20 Initial value for the phase of the seasonal oscillation parameter (cos part)
#' @param alphaT0 Initial value for the infectivity intercept
#' @param InfPrd Infectious period that can be obtained either from the literature or by fitting an SEIRS model to the data
#' @param IncPrd Incubation period that can be obtained either from the literature or by fitting an SEIRS model to the data
#' @param NIterMC Number of MCMC iterations
#' @param NIterMCECM Number of MCECM iterations
#' @param T_cycle The duration of a complete seasonal cycle (e.g., 12 months for an annual cycle)
#' @return
#'
#'   `alphaS` Estimate of alpha S
#'
#'   `BetaCovInf` Estimate of beta vector for the individual level infection covariate
#'
#'   `BetaCovSus` Estimate of beta vector for the areal susceptibility to first infection covariate
#'
#'   `alphaT` Estimate of alpha T
#'
#'   `delta` Estimate of delta
#'
#'   `zeta1` Estimate of zeta1
#'
#'   `zeta2` Estimate of zeta2
#'
#'   `tau1` Estimate of tau
#'
#'   `lambda1` Estimate of lambda
#'
#'   `AIC` AIC of the fitted GDILM SEIRS
#'
#' @export
#' @import MASS
#' @import mvtnorm
#' @import ngspatial
#' @import stats
#'
#' @examples
#' \donttest{
#' data(data)
#' data(adjacency_matrix)
#' SeasEpi_Par_Est(data,adjacency_matrix,2,2,0.5, 0.5, 1, 0.1, 1, 1, 1, 20, 2,0.2,0.2,5)
#' }
#'
SeasEpi_Par_Est=function(data,adjacency_matrix,DimCovInf,DimCovSus,tau0, lambda0, alphaS0, delta0, alphaT0,InfPrd, IncPrd, NIterMC, NIterMCECM,zeta10,zeta20,T_cycle){
if(lambda0>1) stop("The spatial dependence parameter should be restricted to a range between 0 and 1.")
if(lambda0==0) stop("Absence of spatial dependence: This model is designed for scenarios where spatial dependence is present.")
if(delta0<0) stop("The spatial decay parameter must be greater than zero.")
if(NIterMC<2) stop("The number of iterations must exceed 2.")
if(InfPrd<0) stop("The infectious period must be greater than zero.")
if(IncPrd<0) stop("The incubation period must be greater than zero.")
if(DimCovInf<0) stop("Dimensions of the individual infectivity covariate must be greater than zero.")
if(DimCovSus<0) stop("Dimensions of the area-level susceptibility to initial infection covariate must be greater than zero.")
  NTotalpost=nrow(data)
  NTotalGrid=length(unique(data$Label_NC_shape))
  NAllPostPerGrid <- table(data$Label_NC_shape)
  MaxTimePand=max(ceiling(data[,8]))+15
  Lat=data[,1]
  Long=data[,2]
  I=diag(NTotalGrid)
  D=-adjacency_matrix
  diag(D)=colSums(adjacency_matrix)
NLableGrid=as.numeric(as.vector(data[,9]))
  NewLabelGrid=matrix(0,NTotalpost,NTotalGrid)
  for(RHAD in 1:NTotalGrid){
    for(i in 1:NTotalpost){
      if(D[NLableGrid[i],RHAD]!=0){
        NewLabelGrid[i,RHAD]=-1
      }
    }
  }
  Dist=matrix(0,NTotalpost,NTotalpost)
  for(i in 1:NTotalpost){
    for(j in 1:NTotalpost){
      Dist[i,j]=sqrt((Lat[i]-Lat[j])^2+(Long[i]-Long[j])^2)
    }
  }
  Dist=Dist*50
  Pop=data[,3]
  NInf=data[,4]
  phi=mvrnorm(1, rep(0,NTotalGrid), solve(tau0^2*(lambda0*D+(1-lambda0)*I)), tol = 1e-6)
  CovInf=as.matrix(data[,c(6,7)])
  dimnames(   CovInf) <- NULL
  CovSus=as.matrix(data[, c(10, 11)])
  dimnames( CovSus) <- NULL
  BetaCovInf0=rep(1,DimCovInf)
  BetaCovSus0=rep(1,DimCovSus)
  InfPeriod=rep(InfPrd,NTotalpost)
  IncPeriod=rep(IncPrd,NTotalpost)
  ExpoTime=ceiling(data[,8])
  InfTime=ExpoTime+IncPeriod[1]
  is_exposed <- function(i, t, ExpoTime, IncPeriod) {
    ExpoTime[i] <= t && (ExpoTime[i] + IncPeriod[i]) > t && ExpoTime[i] != 0
  }
  F1 <- function(NLableGrid, Dist, alphaS, delta, zeta1, zeta2, T_cycle,
                 i, GridIndic, t, BetaCovInf, BetaCovSus, alphaT) {

    total <- 0

    for (j in 1:NTotalpost) {
      active <- NewLabelGrid[j, GridIndic] != 0 &&
        InfTime[j] <= t &&
        (InfTime[j] + InfPeriod[j]) >= t &&
        InfTime[j] != 0

      if (active) {
        value <- exp(alphaT + CovInf[j, ] %*% BetaCovInf -
                       delta * Dist[i, j] +
                       zeta1 * sin(2 * pi * t / T_cycle) +
                       zeta2 * cos(2 * pi * t / T_cycle))

        if (!is.finite(value)) value <- 0
        total <- total + value
      }
    }

    total * exp(alphaS + CovSus[GridIndic, ] %*% BetaCovSus)
  }

  F2 <- function(NLableGrid, Dist, alphaS, delta, zeta1, zeta2, T_cycle,
                 i, GridIndic, t, BetaCovInf, BetaCovSus, alphaT) {

    total <- rep(0, DimCovInf)
    sin_term <- sin(2 * pi * t / T_cycle)
    cos_term <- cos(2 * pi * t / T_cycle)

    for (j in 1:NTotalpost) {
      active <- NewLabelGrid[j, GridIndic] != 0 &&
        InfTime[j] <= t &&
        (InfTime[j] + InfPeriod[j]) >= t &&
        InfTime[j] != 0

      if (active) {
        val <- as.numeric(exp(alphaT + as.numeric(CovInf[j, ] %*% BetaCovInf) -
                                delta * Dist[i, j] +
                                zeta1 * sin_term +
                                zeta2 * cos_term))
        total <- total + CovInf[j, ] * val
      }
    }

    total * as.numeric(exp(alphaS + as.numeric(CovSus[GridIndic, ] %*% BetaCovSus)))
  }

  F3 <- function(NLableGrid, Dist, alphaS, delta, zeta1, zeta2, T_cycle,
                 i, GridIndic, t, BetaCovInf, BetaCovSus, alphaT) {

    total <- matrix(0, nrow = DimCovInf, ncol = DimCovInf)
    sin_term <- sin(2 * pi * t / T_cycle)
    cos_term <- cos(2 * pi * t / T_cycle)

    for (j in 1:NTotalpost) {
      active <- NewLabelGrid[j, GridIndic] != 0 &&
        InfTime[j] <= t &&
        (InfTime[j] + InfPeriod[j]) >= t &&
        InfTime[j] != 0

      if (active) {
        val <- as.numeric(exp(alphaT + as.numeric(CovInf[j, ] %*% BetaCovInf) -
                                delta * Dist[i, j] +
                                zeta1 * sin_term +
                                zeta2 * cos_term))
        total <- total + CovInf[j, ] %*% t(CovInf[j, ]) * val
      }
    }

    total * as.numeric(exp(alphaS + as.numeric(CovSus[GridIndic, ] %*% BetaCovSus)))
  }

  F4 <- function(NLableGrid, Dist, alphaS, delta, zeta1, zeta2, T_cycle,
                 i, GridIndic, t, BetaCovInf, BetaCovSus, alphaT) {

    total <- 0

    for (j in 1:NTotalpost) {
      active <- NewLabelGrid[j, GridIndic] != 0 &&
        InfTime[j] <= t &&
        (InfTime[j] + InfPeriod[j]) >= t &&
        InfTime[j] != 0

      if (active) {
        val <- exp(alphaT + as.numeric(CovInf[j, ] %*% BetaCovInf)) *
          Dist[i, j]^(-delta) * log(Dist[i, j])
        total <- total + ifelse(is.finite(val), val, 0)
      }
    }

    total * exp(as.numeric(CovSus[GridIndic, ] %*% BetaCovSus) + alphaS)
  }

  F5 <- function(NLableGrid, Dist, alphaS, delta, zeta1, zeta2, T_cycle,
                 i, GridIndic, t, BetaCovInf, BetaCovSus, alphaT) {

    total <- 0

    for (j in 1:NTotalpost) {
      active <- NewLabelGrid[j, GridIndic] != 0 &&
        InfTime[j] <= t &&
        (InfTime[j] + InfPeriod[j]) >= t &&
        InfTime[j] != 0

      if (active) {
        val <- exp(alphaT + as.numeric(CovInf[j, ] %*% BetaCovInf) -
                     delta * Dist[i, j] +
                     zeta1 * sin(2 * pi * t / T_cycle) +
                     zeta2 * cos(2 * pi * t / T_cycle))
        total <- total + ifelse(is.finite(val), val, 0)
      }
    }

    total * exp(as.numeric(CovSus[GridIndic, ] %*% BetaCovSus) + alphaS)
  }


  F6 <- function(NLableGrid, Dist, alphaS, delta, zeta1, zeta2, T_cycle,
                 i, GridIndic, t, BetaCovInf, BetaCovSus, alphaT) {

    total <- 0
    sin_term <- sin(2 * pi * t / T_cycle)

    for (j in 1:NTotalpost) {
      active <- NewLabelGrid[j, GridIndic] != 0 &&
        InfTime[j] <= t &&
        (InfTime[j] + InfPeriod[j]) >= t &&
        InfTime[j] != 0

      if (active) {
        val <- sin_term * exp(alphaT + as.numeric(CovInf[j, ] %*% BetaCovInf) -
                                delta * Dist[i, j] +
                                zeta1 * sin_term +
                                zeta2 * cos(2 * pi * t / T_cycle))
        total <- total + ifelse(is.finite(val), val, 0)
      }
    }

    total * exp(as.numeric(CovSus[GridIndic, ] %*% BetaCovSus) + alphaS)
  }

  F7 <- function(NLableGrid, Dist, alphaS, delta, zeta1, zeta2, T_cycle,
                 i, GridIndic, t, BetaCovInf, BetaCovSus, alphaT) {

    total <- 0
    sin_term <- sin(2 * pi * t / T_cycle)
    sin2 <- sin_term^2
    cos_term <- cos(2 * pi * t / T_cycle)

    for (j in 1:NTotalpost) {
      active <- NewLabelGrid[j, GridIndic] != 0 &&
        InfTime[j] <= t &&
        (InfTime[j] + InfPeriod[j]) >= t &&
        InfTime[j] != 0

      if (active) {
        val <- sin2 * exp(alphaT + as.numeric(CovInf[j, ] %*% BetaCovInf) -
                            delta * Dist[i, j] +
                            zeta1 * sin_term +
                            zeta2 * cos_term)
        total <- total + ifelse(is.finite(val), val, 0)
      }
    }

    total * exp(as.numeric(CovSus[GridIndic, ] %*% BetaCovSus) + alphaS)
  }

  F8 <- function(NLableGrid, Dist, alphaS, delta, zeta1, zeta2, T_cycle,
                 i, GridIndic, t, BetaCovInf, BetaCovSus, alphaT) {

    total <- 0
    cos_term <- cos(2 * pi * t / T_cycle)
    sin_term <- sin(2 * pi * t / T_cycle)

    for (j in 1:NTotalpost) {
      active <- NewLabelGrid[j, GridIndic] != 0 &&
        InfTime[j] <= t &&
        (InfTime[j] + InfPeriod[j]) >= t &&
        InfTime[j] != 0

      if (active) {
        val <- cos_term * exp(alphaT + as.numeric(CovInf[j, ] %*% BetaCovInf) -
                                delta * Dist[i, j] +
                                zeta1 * sin_term +
                                zeta2 * cos_term)
        total <- total + ifelse(is.finite(val), val, 0)
      }
    }

    total * exp(as.numeric(CovSus[GridIndic, ] %*% BetaCovSus) + alphaS)
  }

  F9 <- function(NLableGrid, Dist, alphaS, delta, zeta1, zeta2, T_cycle,
                 i, GridIndic, t, BetaCovInf, BetaCovSus, alphaT) {

    total <- 0
    cos_term <- cos(2 * pi * t / T_cycle)
    sin_term <- sin(2 * pi * t / T_cycle)

    for (j in 1:NTotalpost) {
      active <- NewLabelGrid[j, GridIndic] != 0 &&
        InfTime[j] <= t &&
        (InfTime[j] + InfPeriod[j]) >= t &&
        InfTime[j] != 0

      if (active) {
        val <- (cos_term)^2 * exp(alphaT + as.numeric(CovInf[j, ] %*% BetaCovInf) -
                                    delta * Dist[i, j] +
                                    zeta1 * sin_term +
                                    zeta2 * cos_term)
        total <- total + ifelse(is.finite(val), val, 0)
      }
    }

    total * exp(as.numeric(CovSus[GridIndic, ] %*% BetaCovSus) + alphaS)
  }

  F10 <- function(phi, alphaS, delta, zeta1, zeta2, T_cycle, lambda1, tau1,
                  BetaCovInf, BetaCovSus, alphaT) {

    F11 <- array(0, c(NTotalpost, MaxTimePand, NTotalGrid))
    sin_term <- sin(2 * pi * (1:MaxTimePand) / T_cycle)
    cos_term <- cos(2 * pi * (1:MaxTimePand) / T_cycle)

    for (i in 1:NTotalpost) {
      GridIndic_i <- NLableGrid[i]
      for (t in 1:MaxTimePand) {

        if (!is.na(GridIndic_i)) {

          compute_dx <- function() {
            INDFN <- rep(0, NTotalpost)
            for (j in 1:NTotalpost) {
              if (NewLabelGrid[j, GridIndic_i] != 0 &&
                  InfTime[j] <= t && (InfTime[j] + InfPeriod[j]) >= t &&
                  InfTime[j] != 0) {

                val <- exp(alphaT + as.numeric(CovInf[j, ] %*% BetaCovInf) -
                             delta * Dist[i, j] +
                             zeta1 * sin_term[t] + zeta2 * cos_term[t])
                INDFN[j] <- ifelse(is.finite(val), val, 0)
              }
            }
            sum(INDFN)
          }

          if (ExpoTime[i] > t || ExpoTime[i] == 0) {

            PR <- 1 - exp(-exp(alphaS + as.numeric(CovSus[GridIndic_i, ] %*% BetaCovSus) + phi[GridIndic_i]) * compute_dx())
            F11[i, t, GridIndic_i] <- 1 - PR
          }

          if (is_exposed(i, t, ExpoTime, IncPeriod)) {
            PR <- 1 - exp(-exp(alphaS + as.numeric(CovSus[GridIndic_i, ] %*% BetaCovSus) + phi[GridIndic_i]) * compute_dx())
            F11[i, t, GridIndic_i] <- PR
          }
        }
      }
    }

    F12 <- numeric(NTotalGrid)
    for (GridIndic in 1:NTotalGrid) {
      PP <- sapply(1:NTotalpost, function(i) round(prod(F11[i, , GridIndic][F11[i, , GridIndic] > 0]), 10))
      H <- which(PP != 1)
      F12[GridIndic] <- prod(PP[H][PP[H] > 0])
    }

    return(F12)
  }

  alphaS=alphaS0
  delta=delta0
  zeta1=zeta10
  zeta2=zeta20
  tau1=tau0
  lambda1=lambda0
  BetaCovInf=BetaCovInf0
  BetaCovSus=BetaCovSus0
  alphaT=alphaT0

  MCECM=function(NLableGrid,Dist,alphaS,delta,zeta1,zeta2,T_cycle,lambda1,tau1,BetaCovInf,BetaCovSus,alphaT){
    PSTCD <- matrix(0, NIterMC + 1, NTotalGrid)

    PSTCD[1, ] <- mvrnorm(1, rep(0, NTotalGrid), solve(tau1^2 * (lambda1 * D + (1 - lambda1) * I)), tol = 1e-6)

    CPU4 <- numeric(NIterMC)
    CPU3 <- numeric(NIterMC)

    for (L in 2:NIterMC) {
      phi <- mvrnorm(1, rep(0, NTotalGrid), solve(tau1^2 * (lambda1 * D + (1 - lambda1) * I)), tol = 1e-6)

      CPU1 <- F10(phi, alphaS, delta, zeta1, zeta2, T_cycle, lambda1, tau1, BetaCovInf, BetaCovSus, alphaT)
      CPU2 <- F10(PSTCD[L - 1, ], alphaS, delta, zeta1, zeta2, T_cycle, lambda1, tau1, BetaCovInf, BetaCovSus, alphaT)

      CPU3[L] <- min(1, prod(CPU1 / CPU2))
      CPU4[L] <- runif(1)

      PSTCD[L, ] <- if (CPU4[L] < CPU3[L]) phi else PSTCD[L - 1, ]
    }


    F13 <- function(PSTCD, GridIndic) {
      return(mean(exp(PSTCD[1:NIterMC, GridIndic])))
    }

    F14 <- function(NLableGrid, PSTCD, Dist, alphaS, delta, zeta1, zeta2, T_cycle,
                    lambda1, i, GridIndic, t, BetaCovInf, BetaCovSus, alphaT, L) {

      FN <- rep(0, NTotalpost)

      for (j in 1:NTotalpost) {
        if (NewLabelGrid[j, GridIndic] != 0 &&
            InfTime[j] <= t && (InfTime[j] + InfPeriod[j]) >= t && InfTime[j] != 0) {
          FN[j] <- exp(alphaT + CovInf[j, ] %*% BetaCovInf - delta * Dist[i, j] +
                         zeta1 * sin(2 * pi * t / T_cycle) + zeta2 * cos(2 * pi * t / T_cycle))
        }
      }

      FN[!is.finite(FN)] <- 0

      PRSEC <- 1 - exp(-exp(alphaS + CovSus[GridIndic, ] %*% BetaCovSus + PSTCD[L, GridIndic]) * sum(FN))


      return(ifelse(PRSEC == 0, 0, (1 - PRSEC) / PRSEC * exp(PSTCD[L, GridIndic])))
    }

    F15 <- function(NLableGrid, PSTCD, Dist, alphaS, delta, zeta1, zeta2, T_cycle,
                    lambda1, i, GridIndic, t, BetaCovInf, BetaCovSus, alphaT, L) {

      FN <- rep(0, NTotalpost)

      for (j in 1:NTotalpost) {
        if (NewLabelGrid[j, GridIndic] != 0 &&
            InfTime[j] <= t && (InfTime[j] + InfPeriod[j]) >= t && InfTime[j] != 0) {
          FN[j] <- exp(alphaT + CovInf[j, ] %*% BetaCovInf - delta * Dist[i, j] +
                         zeta1 * sin(2 * pi * t / T_cycle) + zeta2 * cos(2 * pi * t / T_cycle))
        }
      }

      FN[!is.finite(FN)] <- 0

      PRSEC <- 1 - exp(-exp(alphaS + CovSus[GridIndic, ] %*% BetaCovSus + PSTCD[L, GridIndic]) * sum(FN))

      return(ifelse(PRSEC == 0, 0, (1 - PRSEC) / PRSEC^2 * exp(2 * PSTCD[L, GridIndic])))
    }

    F16 <- numeric(MaxTimePand)
    F17 <- numeric(MaxTimePand)
    F18 <- numeric(MaxTimePand)

    for (t in 1:MaxTimePand) {

      F19 <- numeric(NTotalpost)
      for (i in 1:NTotalpost) {
        GridIndic <- NLableGrid[i]
        if (ExpoTime[i] > t || ExpoTime[i] == 0) {
          F19[i] <- -F1(NLableGrid, Dist, alphaS, delta, zeta1, zeta2, T_cycle,
                        i, GridIndic, t, BetaCovInf, BetaCovSus, alphaT) *
            F13(PSTCD, GridIndic)
        }
      }
      F16[t] <- sum(F19)

      F20 <- numeric(NTotalpost)
      for (i in 1:NTotalpost) {
        GridIndic <- NLableGrid[i]
        if (is_exposed(i, t, ExpoTime, IncPeriod)) {
          F21 <- numeric(NIterMC)
          for (L in 1:NIterMC) {
            F21[L] <- F14(NLableGrid, PSTCD, Dist, alphaS, delta, zeta1, zeta2, T_cycle,
                          lambda1, i, GridIndic, t, BetaCovInf, BetaCovSus, alphaT, L)
          }
          F20[i] <- F1(NLableGrid, Dist, alphaS, delta, zeta1, zeta2, T_cycle,
                       i, GridIndic, t, BetaCovInf, BetaCovSus, alphaT) * mean(F21)
        }
      }
      F17[t] <- sum(F20)

      F22 <- numeric(NTotalpost)
      for (i in 1:NTotalpost) {
        GridIndic <- NLableGrid[i]
        if (is_exposed(i, t, ExpoTime, IncPeriod)) {
          F23 <- numeric(NIterMC)
          for (L in 1:NIterMC) {
            F23[L] <- F15(NLableGrid, PSTCD, Dist, alphaS, delta, zeta1, zeta2, T_cycle,
                          lambda1, i, GridIndic, t, BetaCovInf, BetaCovSus, alphaT, L)
          }
          F22[i] <- - (F1(NLableGrid, Dist, alphaS, delta, zeta1, zeta2, T_cycle,
                          i, GridIndic, t, BetaCovInf, BetaCovSus, alphaT))^2 * mean(F23)
        }
      }
      F18[t] <- sum(F22)
    }

    EstAlphaS <- alphaS - (sum(F16) + sum(F17, na.rm = TRUE)) / (sum(F16) + sum(F17, na.rm = TRUE) + sum(F18, na.rm = TRUE))

    F24=array(0,c(DimCovSus,1,MaxTimePand))
    for(t in 1:MaxTimePand){
      F25=array(0,c(DimCovSus,1,NTotalpost))
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if (NLableGrid[i]==GridIndic){
            if(ExpoTime[i]>t|ExpoTime[i]==0){
              F25[,,i]=-CovSus[GridIndic,]*as.numeric(F1(NLableGrid,Dist,EstAlphaS,delta,zeta1,zeta2,T_cycle,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT)*F13(PSTCD,GridIndic))
            }
          }
        }
      }
      F24[,,t]=apply(F25,c(1,2),sum)
    }

    F26=array(0,c(DimCovSus,1,MaxTimePand))
    for(t in 1:MaxTimePand){
      F27=array(0,c(DimCovSus,1,NTotalpost))
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if(NLableGrid[i]==GridIndic){
            if(is_exposed(i, t, ExpoTime, IncPeriod)){
              F28=c()
              for(L in 1:NIterMC){
                F28[L]=F14(NLableGrid,PSTCD,Dist,alphaS,delta,zeta1,zeta2,T_cycle,lambda1,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT,L)
              }
              F27[,,i]=CovSus[GridIndic,]*as.numeric(F1(NLableGrid,Dist,EstAlphaS,delta,zeta1,zeta2,T_cycle,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT)*mean(F28))
            }
          }
        }
      }
      F26[,,t]=apply(F27,c(1,2),sum,na.rm=T)
    }

    F29=array(0,c(DimCovSus,DimCovSus,MaxTimePand))
    for(t in 1:MaxTimePand){
      F30=array(0,c(DimCovSus,DimCovSus,NTotalpost))
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if (NLableGrid[i]==GridIndic){
            if(ExpoTime[i]>t|ExpoTime[i]==0){
              F30[,,i]=-CovSus[GridIndic,]%*%t(CovSus[GridIndic,])*as.numeric(F1(NLableGrid,Dist,EstAlphaS,delta,zeta1,zeta2,T_cycle,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT)*F13(PSTCD,GridIndic))
            }
          }
        }
      }
      F29[,,t]=apply(F30,c(1,2),sum)
    }

    F31=array(0,c(DimCovSus,DimCovSus,MaxTimePand))
    for(t in 1:MaxTimePand){
      F32=array(0,c(DimCovSus,DimCovSus,NTotalpost))
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if(NLableGrid[i]==GridIndic){
            if(is_exposed(i, t, ExpoTime, IncPeriod)){
              F33=c()
              for(L in 1:NIterMC){
                F33[L]=F14(NLableGrid,PSTCD,Dist,alphaS,delta,zeta1,zeta2,T_cycle,lambda1,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT,L)
              }
              F32[,,i]=CovSus[GridIndic,]%*%t(CovSus[GridIndic,])*as.numeric(F1(NLableGrid,Dist,EstAlphaS,delta,zeta1,zeta2,T_cycle,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT)*mean(F33))
            }
          }
        }
      }
      F31[,,t]=apply(F32,c(1,2),sum)
    }


    F34=array(0,c(DimCovSus,DimCovSus,MaxTimePand))
    for(t in 1:MaxTimePand){
      F35=array(0,c(DimCovSus,DimCovSus,NTotalpost))
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if (NLableGrid[i]==GridIndic){
            if(is_exposed(i, t, ExpoTime, IncPeriod)){
              F36=c()
              for(L in 1:NIterMC){
                F36[L]=F15(NLableGrid,PSTCD,Dist,alphaS,delta,zeta1,zeta2,T_cycle,lambda1,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT,L)
              }
              F35[,,i]=-1*CovSus[GridIndic,]%*%t(CovSus[GridIndic,])*as.numeric((F1(NLableGrid,Dist,EstAlphaS,delta,zeta1,zeta2,T_cycle,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT))^2*mean(F36))
            }
          }
        }
      }
      F34[,,t]=apply(F35,c(1,2),sum,na.rm=T)
    }

    F37=apply(F29,c(1,2),sum)+apply(F31,c(1,2),sum,na.rm=T)+apply(F34,c(1,2),sum)

    epsilon <- 1e-3
    if (det(F37) < epsilon) {
      F37 <- F37 + diag(epsilon, nrow(F37))
    }

    EstBetaCovSus=BetaCovSus-solve(F37)%*%(apply(F24,c(1,2),sum)+apply(F26,c(1,2),sum))


    F38=rep(0,MaxTimePand)
    for(t in 1:MaxTimePand){
      F39=rep(0,NTotalpost)
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if (NLableGrid[i]==GridIndic){
            if(ExpoTime[i]>t|ExpoTime[i]==0){
              F39[i]=-as.numeric(F1(NLableGrid,Dist,EstAlphaS,delta,zeta1,zeta2,T_cycle,i,GridIndic,t,BetaCovInf,EstBetaCovSus,alphaT)*F13(PSTCD,GridIndic))
            }
          }
        }
      }
      F38[t]=sum(F39)
    }

    F50=rep(0,MaxTimePand)
    for(t in 1:MaxTimePand){
      F51=rep(0,NTotalpost)
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if(NLableGrid[i]==GridIndic){
            if(is_exposed(i, t, ExpoTime, IncPeriod)){
              F52=c()
              for(L in 1:NIterMC){
                F52[L]=F14(NLableGrid,PSTCD,Dist,alphaS,delta,zeta1,zeta2,T_cycle,lambda1,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT,L)
              }
              F51[i]=as.numeric(F1(NLableGrid,Dist,EstAlphaS,delta,zeta1,zeta2,T_cycle,i,GridIndic,t,BetaCovInf,EstBetaCovSus,alphaT)*mean(F52))
            }
          }
        }
      }
      F50[t]=sum(F51)
    }

    F53=rep(0,MaxTimePand)
    for(t in 1:MaxTimePand){
      F54=rep(0,NTotalpost)
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if (NLableGrid[i]==GridIndic){
            if(is_exposed(i, t, ExpoTime, IncPeriod)){
              F55=c()
              for(L in 1:NIterMC){
                F55[L]=F15(NLableGrid,PSTCD,Dist,alphaS,delta,zeta1,zeta2,T_cycle,lambda1,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT,L)
              }
              F54[i]=-1*as.numeric((F1(NLableGrid,Dist,EstAlphaS,delta,zeta1,zeta2,T_cycle,i,GridIndic,t,BetaCovInf,EstBetaCovSus,alphaT))^2*mean(F55))
            }
          }
        }
      }
      F53[t]=sum(F54)
    }

    EstAlphaT=alphaT-(sum(F38)+sum(F50,na.rm=T))/(sum(F38)+sum(F50,na.rm=T)+sum(F53,na.rm=T))

    F56=array(0,c(DimCovInf,1,MaxTimePand))
    for(t in 1:MaxTimePand){
      F57=array(0,c(DimCovInf,1,NTotalpost))
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if (NLableGrid[i]==GridIndic){
            if(ExpoTime[i]>t|ExpoTime[i]==0){
              F57[,,i]=-F2(NLableGrid,Dist,EstAlphaS,delta,zeta1,zeta2,T_cycle,i,GridIndic,t,BetaCovInf,EstBetaCovSus,EstAlphaT)*F13(PSTCD,GridIndic)
            }
          }
        }
      }
      F56[,,t]=apply(F57,c(1,2),sum,na.rm=T)
    }

    F58=array(0,c(DimCovInf,1,MaxTimePand))
    for(t in 1:MaxTimePand){
      F59=array(0,c(DimCovInf,1,NTotalpost))
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if(NLableGrid[i]==GridIndic){
            if(is_exposed(i, t, ExpoTime, IncPeriod)){
              F60=c()
              for(L in 1:NIterMC){
                F60[L]=F14(NLableGrid,PSTCD,Dist,alphaS,delta,zeta1,zeta2,T_cycle,lambda1,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT,L)
              }
              F59[,,i]=F2(NLableGrid,Dist,EstAlphaS,delta,zeta1,zeta2,T_cycle,i,GridIndic,t,BetaCovInf,EstBetaCovSus,EstAlphaT)*as.numeric(mean(F60))
            }
          }
        }
      }
      F58[,,t]=apply(F59,c(1,2),sum,na.rm=T)
    }

    Q1=array(0,c(DimCovInf,DimCovInf,MaxTimePand))
    for(t in 1:MaxTimePand){
      Q2=array(0,c(DimCovInf,DimCovInf,NTotalpost))
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if (NLableGrid[i]==GridIndic){
            if(ExpoTime[i]>t|ExpoTime[i]==0){
              Q2[,,i]=-F3(NLableGrid,Dist,EstAlphaS,delta,zeta1,zeta2,T_cycle,i,GridIndic,t,BetaCovInf,EstBetaCovSus,EstAlphaT)*as.numeric(F13(PSTCD,GridIndic))
            }
          }
        }
      }
      Q1[,,t]=apply(Q2,c(1,2),sum,na.rm=T)
    }

    Q3=array(0,c(DimCovInf,DimCovInf,MaxTimePand))
    for(t in 1:MaxTimePand){
      Q4=array(0,c(DimCovInf,DimCovInf,NTotalpost))
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if(NLableGrid[i]==GridIndic){
            if(is_exposed(i, t, ExpoTime, IncPeriod)){

              Q5=c()
              for(L in 1:NIterMC){
                Q5[L]=F14(NLableGrid,PSTCD,Dist,alphaS,delta,zeta1,zeta2,T_cycle,lambda1,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT,L)
              }
              Q4[,,i]=F3(NLableGrid,Dist,EstAlphaS,delta,zeta1,zeta2,T_cycle,i,GridIndic,t,BetaCovInf,EstBetaCovSus,EstAlphaT)*as.numeric(mean(Q5))
            }
          }
        }
      }
      Q3[,,t]=apply(Q4,c(1,2),sum,na.rm=T)
    }


    Q6=array(0,c(DimCovInf,DimCovInf,MaxTimePand))
    for(t in 1:MaxTimePand){
      Q8=array(0,c(DimCovInf,DimCovInf,NTotalpost))
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if (NLableGrid[i]==GridIndic){
            if(is_exposed(i, t, ExpoTime, IncPeriod)){
              Q7=c()
              for(L in 1:NIterMC){
                Q7[L]=F15(NLableGrid,PSTCD,Dist,alphaS,delta,zeta1,zeta2,T_cycle,lambda1,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT,L)
              }
              Q8[,,i]=-1*F2(NLableGrid,Dist,EstAlphaS,delta,zeta1,zeta2,T_cycle,i,GridIndic,t,BetaCovInf,EstBetaCovSus,EstAlphaT)%*%t(F2(NLableGrid,Dist,EstAlphaS,delta,zeta1,zeta2,T_cycle,i,GridIndic,t,BetaCovInf,EstBetaCovSus,EstAlphaT))*as.numeric(mean(Q7))
            }
          }
        }
      }
      Q6[,,t]=apply(Q8,c(1,2),sum,na.rm=T)
    }

    EndB5=apply(Q1,c(1,2),sum,na.rm=T)+apply(Q3,c(1,2),sum,na.rm=T)+apply(Q6,c(1,2),sum,na.rm=T)
    EstBetaCovInf=BetaCovInf-solve(EndB5)%*%(apply(F56,c(1,2),sum,na.rm=T)+apply(F58,c(1,2),sum,na.rm=T))
    Q9=rep(0,MaxTimePand)
    for(t in 1:MaxTimePand){
      Q10=rep(0,NTotalpost)
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if(NLableGrid[i]==GridIndic){
            if(ExpoTime[i]>t|ExpoTime[i]==0){
              Q10[i]=as.numeric(F4(NLableGrid,Dist,EstAlphaS,delta,zeta1,zeta2,T_cycle,i,GridIndic,t,EstBetaCovInf,EstBetaCovSus,EstAlphaT)*F13(PSTCD,GridIndic))
            }
          }
        }
      }
      Q9[t]=sum(Q10,na.rm=T)
    }

    Q11=rep(0,MaxTimePand)
    for(t in 1:MaxTimePand){
      Q12=rep(0,NTotalpost)
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if(NLableGrid[i]==GridIndic){
            if(is_exposed(i, t, ExpoTime, IncPeriod)){
              Q13=c()
              for(L in 1:NIterMC){
                Q13[L]=F14(NLableGrid,PSTCD,Dist,alphaS,delta,zeta1,zeta2,T_cycle,lambda1,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT,L)
              }
              Q12[i]=-as.numeric(F4(NLableGrid,Dist,EstAlphaS,delta,zeta1,zeta2,T_cycle,i,GridIndic,t,EstBetaCovInf,EstBetaCovSus,EstAlphaT)*mean(Q13))
            }
          }
        }
      }
      Q11[t]=sum(Q12)
    }


    Q14=rep(0,MaxTimePand)
    for(t in 1:MaxTimePand){
      Q15=rep(0,NTotalpost)
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if(NLableGrid[i]==GridIndic){
            if(ExpoTime[i]>t|ExpoTime[i]==0){
              Q15[i]=-as.numeric(F5(NLableGrid,Dist,EstAlphaS,delta,zeta1,zeta2,T_cycle,i,GridIndic,t,EstBetaCovInf,EstBetaCovSus,EstAlphaT)*F13(PSTCD,GridIndic))
            }
          }
        }
      }
      Q14[t]=sum(Q15)
    }

    Q16=rep(0,MaxTimePand)
    for(t in 1:MaxTimePand){
      Q17=rep(0,NTotalpost)
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if(NLableGrid[i]==GridIndic){
            if(is_exposed(i, t, ExpoTime, IncPeriod)){
              Q18=c()
              for(L in 1:NIterMC){
                Q18[L]=F14(NLableGrid,PSTCD,Dist,alphaS,delta,zeta1,zeta2,T_cycle,lambda1,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT,L)
              }
              Q19=c()
              for(L in 1:NIterMC){
                Q19[L]=F15(NLableGrid,PSTCD,Dist,alphaS,delta,zeta1,zeta2,T_cycle,lambda1,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT,L)
              }
              Q17[i]=as.numeric(F5(NLableGrid,Dist,EstAlphaS,delta,zeta1,zeta2,T_cycle,i,GridIndic,t,EstBetaCovInf,EstBetaCovSus,EstAlphaT)*mean(Q18))-1*as.numeric((F4(NLableGrid,Dist,EstAlphaS,delta,zeta1,zeta2,T_cycle,i,GridIndic,t,EstBetaCovInf,EstBetaCovSus,EstAlphaT))^2*mean(Q19))
            }
          }
        }
      }
      Q16[t]=sum(Q17)
    }

    Estdelta=delta-(sum(Q9,na.rm=T)+sum(Q11,na.rm=T))/(sum(Q14)+sum(Q16,na.rm=T))
    Q20=rep(0,MaxTimePand)
    for(t in 1:MaxTimePand){
      Q21=rep(0,NTotalpost)
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if(NLableGrid[i]==GridIndic){
            if(ExpoTime[i]>t|ExpoTime[i]==0){
              Q21[i]=as.numeric(F6(NLableGrid,Dist,EstAlphaS,Estdelta,zeta1,zeta2,T_cycle,i,GridIndic,t,EstBetaCovInf,EstBetaCovSus,EstAlphaT)*F13(PSTCD,GridIndic))
            }
          }
        }
      }
      Q20[t]=sum(Q21,na.rm=T)
    }

    Q22=rep(0,MaxTimePand)
    for(t in 1:MaxTimePand){
      Q23=rep(0,NTotalpost)
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if(NLableGrid[i]==GridIndic){
            if(is_exposed(i, t, ExpoTime, IncPeriod)){
              Q24=c()
              for(L in 1:NIterMC){
                Q24[L]=F14(NLableGrid,PSTCD,Dist,alphaS,delta,zeta1,zeta2,T_cycle,lambda1,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT,L)
              }
              Q23[i]=-as.numeric(F6(NLableGrid,Dist,EstAlphaS,Estdelta,zeta1,zeta2,T_cycle,i,GridIndic,t,EstBetaCovInf,EstBetaCovSus,EstAlphaT)*mean(Q24))
            }
          }
        }
      }
      Q22[t]=sum(Q23)
    }


    Q25=rep(0,MaxTimePand)
    for(t in 1:MaxTimePand){
      Q26=rep(0,NTotalpost)
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if(NLableGrid[i]==GridIndic){
            if(ExpoTime[i]>t|ExpoTime[i]==0){
              Q26[i]=-as.numeric(F7(NLableGrid,Dist,EstAlphaS,Estdelta,zeta1,zeta2,T_cycle,i,GridIndic,t,EstBetaCovInf,EstBetaCovSus,EstAlphaT)*F13(PSTCD,GridIndic))
            }
          }
        }
      }
      Q25[t]=sum(Q26)
    }

    Q27=rep(0,MaxTimePand)
    for(t in 1:MaxTimePand){
      Q28=rep(0,NTotalpost)
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if(NLableGrid[i]==GridIndic){
            if(is_exposed(i, t, ExpoTime, IncPeriod)){
              Q29=c()
              for(L in 1:NIterMC){
                Q29[L]=F14(NLableGrid,PSTCD,Dist,alphaS,delta,zeta1,zeta2,T_cycle,lambda1,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT,L)
              }
              Q30=c()
              for(L in 1:NIterMC){
                Q30[L]=F15(NLableGrid,PSTCD,Dist,alphaS,delta,zeta1,zeta2,T_cycle,lambda1,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT,L)
              }
              Q28[i]=as.numeric(F7(NLableGrid,Dist,EstAlphaS,Estdelta,zeta1,zeta2,T_cycle,i,GridIndic,t,EstBetaCovInf,EstBetaCovSus,EstAlphaT)*mean(Q29))-1*as.numeric((F6(NLableGrid,Dist,EstAlphaS,Estdelta,zeta1,zeta2,T_cycle,i,GridIndic,t,EstBetaCovInf,EstBetaCovSus,EstAlphaT))^2*mean(Q30))
            }
          }
        }
      }
      Q27[t]=sum(Q28)
    }


    Estzeta1=zeta1-(sum(Q20,na.rm=T)+sum(Q22,na.rm=T))/(sum(Q25)+sum(Q27,na.rm=T))


    Q31=rep(0,MaxTimePand)
    for(t in 1:MaxTimePand){
      Q32=rep(0,NTotalpost)
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if(NLableGrid[i]==GridIndic){
            if(ExpoTime[i]>t|ExpoTime[i]==0){
              Q32[i]=as.numeric(F8(NLableGrid,Dist,EstAlphaS,Estdelta,Estzeta1,zeta2,T_cycle,i,GridIndic,t,EstBetaCovInf,EstBetaCovSus,EstAlphaT)*F13(PSTCD,GridIndic))
            }
          }
        }
      }
      Q31[t]=sum(Q32,na.rm=T)
    }


    Q33=rep(0,MaxTimePand)
    for(t in 1:MaxTimePand){
      Q34=rep(0,NTotalpost)
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if(NLableGrid[i]==GridIndic){
            if(is_exposed(i, t, ExpoTime, IncPeriod)){
              Q35=c()
              for(L in 1:NIterMC){
                Q35[L]=F14(NLableGrid,PSTCD,Dist,alphaS,delta,zeta1,zeta2,T_cycle,lambda1,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT,L)
              }
              Q34[i]=-as.numeric(F8(NLableGrid,Dist,EstAlphaS,Estdelta,Estzeta1,zeta2,T_cycle,i,GridIndic,t,EstBetaCovInf,EstBetaCovSus,EstAlphaT)*mean(Q35))
            }
          }
        }
      }
      Q33[t]=sum(Q34)
    }

    Q36=rep(0,MaxTimePand)
    for(t in 1:MaxTimePand){
      Q37=rep(0,NTotalpost)
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if(NLableGrid[i]==GridIndic){
            if(ExpoTime[i]>t|ExpoTime[i]==0){
              Q37[i]=-as.numeric(F9(NLableGrid,Dist,EstAlphaS,Estdelta,Estzeta1,zeta2,T_cycle,i,GridIndic,t,EstBetaCovInf,EstBetaCovSus,EstAlphaT)*F13(PSTCD,GridIndic))
            }
          }
        }
      }
      Q36[t]=sum(Q37)
    }

    Q38=rep(0,MaxTimePand)
    for(t in 1:MaxTimePand){
      Q39=rep(0,NTotalpost)
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if(NLableGrid[i]==GridIndic){
            if(is_exposed(i, t, ExpoTime, IncPeriod)){
              Q40=c()
              for(L in 1:NIterMC){
                Q40[L]=F14(NLableGrid,PSTCD,Dist,alphaS,delta,zeta1,zeta2,T_cycle,lambda1,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT,L)
              }
              Q41=c()
              for(L in 1:NIterMC){
                Q41[L]=F15(NLableGrid,PSTCD,Dist,alphaS,delta,zeta1,zeta2,T_cycle,lambda1,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT,L)
              }
              Q39[i]=as.numeric(F9(NLableGrid,Dist,EstAlphaS,Estdelta,Estzeta1,zeta2,T_cycle,i,GridIndic,t,EstBetaCovInf,EstBetaCovSus,EstAlphaT)*mean(Q40))-1*as.numeric((F8(NLableGrid,Dist,EstAlphaS,Estdelta,Estzeta1,zeta2,T_cycle,i,GridIndic,t,EstBetaCovInf,EstBetaCovSus,EstAlphaT))^2*mean(Q41))
            }
          }
        }
      }
      Q38[t]=sum(Q39)
    }
    Estzeta2=zeta2-(sum(Q31,na.rm=T)+sum(Q33,na.rm=T))/(sum(Q36)+sum(Q38,na.rm=T))

    LGK=function(par){
      lambda1=par[[1]]
      tau1=par[[2]]
      LG=c()
      for(L in 1:NIterMC){
        LG[L]=dmvnorm(PSTCD[L,], rep(0,NTotalGrid), solve(tau1^2*(as.numeric(lambda1)*D+(1-as.numeric(lambda1))*I)), log = t)
      }
      -mean(LG)
    }

    OPTM=optim(c(lambda1,tau1),LGK)
    OPTMest=OPTM$par
    Gammauest=OPTMest[1]
    SigmmaUest=OPTMest[2]

    result=list(PSTCD=PSTCD,BetaCovInf=EstBetaCovInf,BetaCovSus=EstBetaCovSus,Uhat=OPTMest,alphaS=EstAlphaS,alphaT=EstAlphaT,delta=Estdelta,zeta1=Estzeta1,zeta2=Estzeta2,tau1=SigmmaUest,lambda1=Gammauest)
    result
  }

  LCB=numeric()
  Loglik=function(NLableGrid,PST,Dist,alphaS,delta,zeta1,zeta2,T_cycle,lambda1,tau1,BetaCovInf,BetaCovSus,alphaT){

    FNF=function(PST,GridIndic){
      INDCT=c()
      for(L in 1:NIterMC){
        INDCT[L]=exp(PST[L,GridIndic])
      }
      return(mean(INDCT))
    }

    FNF2=function(NLableGrid,PST,Dist,alphaS,delta,zeta1,zeta2,T_cycle,lambda1,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT,L){
      FN=rep(0,NTotalpost)
      for(j in 1:NTotalpost){
        if (NewLabelGrid[j,GridIndic]!=0){
          if(InfTime[j]<=t & (InfTime[j]+InfPeriod[j])>=t & InfTime[j]!=0){
            FN[j]=exp(alphaT+CovInf[j,]%*%BetaCovInf-delta*Dist[i,j]+zeta1*sin(2*pi*t/T_cycle)+zeta2*cos(2*pi*t/T_cycle))
          }
        }
      }

      PRSEC=1-exp(-exp(alphaS+CovSus[GridIndic,]%*%BetaCovSus+PST[L,GridIndic])*sum(replace(FN,is.infinite(FN),0)))
      if(PRSEC==0){SMean2L=0}
      if(PRSEC!=0){
      }
      return(log(PRSEC))
    }

    FK1=rep(0,MaxTimePand)
    for(t in 1:MaxTimePand){
      FK2=rep(0,NTotalpost)
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if (NLableGrid[i]==GridIndic){
            if(ExpoTime[i]>t|ExpoTime[i]==0){
              FK2[i]=-as.numeric(F1(NLableGrid,Dist,alphaS,delta,zeta1,zeta2,T_cycle,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT)*FNF(PST,GridIndic))
            }
          }
        }
      }
      FK1[t]=sum(FK2)
    }

    FK3=rep(0,MaxTimePand)
    for(t in 1:MaxTimePand){
      FK4=rep(0,NTotalpost)
      for(i in 1:NTotalpost){
        for(GridIndic in 1:NTotalGrid){
          if (NLableGrid[i]==GridIndic){
            if(is_exposed(i, t, ExpoTime, IncPeriod)){
              FK5=c()
              for(L in 1:NIterMC){
                FK5[L]=FNF2(NLableGrid,PST,Dist,alphaS,delta,zeta1,zeta2,T_cycle,lambda1,i,GridIndic,t,BetaCovInf,BetaCovSus,alphaT,L)
              }
              FK4[i]=as.numeric(mean(FK5))
            }
          }
        }
      }
      FK3[t]=sum(FK4)
    }
    LG=c()
    for(L in 1:NIterMC){
      LG[L]=dmvnorm(PST[L,], rep(0,NTotalGrid), solve(tau1^2*(as.numeric(lambda1)*D+(1-as.numeric(lambda1))*I)), log = T)
    }

    return(mean(LG)+sum(FK3)+sum(FK1))
  }

  MCECMest0=MCECM(NLableGrid,Dist,alphaS0,delta0,zeta10,zeta20,T_cycle,lambda0,tau0,BetaCovInf0,BetaCovSus0,alphaT0)
  alphaS=MCECMest0$alphaS
  alphaS
  delta=MCECMest0$delta
  delta
  zeta1=MCECMest0$zeta1
  zeta1
  zeta2=MCECMest0$zeta2
  zeta2
  lambda1=MCECMest0$lambda1
  lambda1
  tau1=MCECMest0$tau1
  tau1
  BetaCovInf=MCECMest0$BetaCovInf
  BetaCovInf
  BetaCovSus=MCECMest0$BetaCovSus
  BetaCovSus
  alphaT=MCECMest0$alphaT
  alphaT
  Uhat=MCECMest0$Uhat
  Uhat
  PST=MCECMest0$PSTCD
  PST
  QQ <- list()
  AIC <- numeric()
  QFN <- numeric()
  TLRNC <- 0.1
  for (NDC in 1:NIterMCECM) {

    MCECMest <- MCECM(NLableGrid, Dist, alphaS, delta, zeta1,zeta2,T_cycle,lambda1, tau1, BetaCovInf, BetaCovSus, alphaT)

    alphaS <- MCECMest$alphaS
    BetaCovInf <- MCECMest$BetaCovInf
    BetaCovSus <- MCECMest$BetaCovSus
    delta <- MCECMest$delta
    zeta1<- MCECMest$zeta1
    zeta2<- MCECMest$zeta2
    lambda1 <- MCECMest$lambda1
    tau1 <- MCECMest$tau1
    alphaT <- MCECMest$alphaT
    Uhat <- MCECMest$Uhat
    PSTCD <- MCECMest$PSTCD

    LCB[NDC] <- Loglik(NLableGrid, PSTCD, Dist, alphaS, delta, zeta1,zeta2,T_cycle,lambda1, tau1, BetaCovInf, BetaCovSus,alphaT)

    AIC[NDC] <- -2 * LCB[NDC] + 11

    out1 <- list(
      alphaS = alphaS,
      BetaCovInf = BetaCovInf,
      BetaCovSus = BetaCovSus,
      alphaT = alphaT,
      delta = delta,
      zeta1 = zeta1,
      zeta2 = zeta2,
      tau1 = tau1,
      lambda1 = lambda1,
      AIC = AIC[NDC]
    )

    QQ[[NDC]] <- c(alphaS, BetaCovInf, BetaCovSus,alphaT, delta, zeta1,zeta2,tau1, lambda1)

    if (NDC > 1) {
      QFN[NDC] <- sqrt(sum((QQ[[NDC]] - QQ[[NDC-1]])^2))
    } else {
      QFN[NDC] <- Inf
    }

    if (QFN[NDC] < TLRNC) {
      message("MCECM Converged at iteration ", NDC, " with parameter change: ", QFN[NDC])
      break
    }

    if (NDC %% 10 == 0) {
      message("Iteration ", NDC, ": Parameter change = ", QFN[NDC])
    }
  }

  out1

}

