#' Generating pixellated populations, and population frames
#' 
#' 
#' `r lifecycle::badge("experimental")`
#' 
#' Functions for generating pixellated population information and 
#' population frames at the `area` and `subarea` levels.  
#' The `area` and `subarea` levels can be thought of as big 
#' regions and little regions, where areas can be partitioned into 
#' unique sets of subareas. For example, Admin-1 and Admin-2 
#' areas might be areas and subareas respectively. The population totals are either 
#' tabulated at the area x urban/rural level, the subarea x urban/rural 
#' level, or at the pixel level of a specified resolution. Totals 
#' are calculated using population density information, shapefiles, 
#' and, possibly, preexisting population frames at different 
#' areal levels. Note that area names should each be unique, and similarly for 
#' subarea names.
#' 
#' @describeIn makePopIntegrationTab Generate pixellated `grid` of coordinates (both longitude/latitude and east/north) 
#' over spatial domain of the given resolution with associated population totals, areas, subareas, 
#' and urban/rural levels. For very small areas that might not 
#' otherwise have a grid point in them, a custom integration point is added at their 
#' centroid. Sets urbanicity classifications by thresholding input population density raster 
#' using area and subarea population tables, and generates area and subarea population 
#' tables from population density information if not already given. Can be used for integrating 
#' predictions from the given coordinates to area and subarea levels using population weights.
#' 
#' @param pop.mat Pixellated grid data frame with variables `area` and `pop` such as that 
#' generated by \code{\link{makePopIntegrationTab}}
#' @param poppa data.frame of population per area separated by urban/rural. If `poppsub` 
#'   is not included, this is used for normalization of populations associated with 
#'   population integration points. Contains variables:
#' \describe{
#'   \item{area}{name of area}
#'   \item{popUrb}{total urban (general) population of area}
#'   \item{popRur}{total rural (general) population of area}
#'   \item{popTotal}{total (general) population of area}
#'   \item{pctUrb}{percentage of population in the area that is urban (between 0 and 100)}
#' }
#' @param poppsub data.frame of population per subarea separated by 
#' urban/rural using for population normalization or urbanicity 
#' classification. Often based on extra fine scale population density grid. 
#' Has variables:
#' \describe{
#'   \item{subarea}{name of subarea}
#'   \item{area}{name of area}
#'   \item{popUrb}{total urban (general) population of subarea}
#'   \item{popRur}{total rural (general) population of subarea}
#'   \item{popTotal}{total (general) population of subarea}
#'   \item{pctUrb}{percentage of population in the subarea that is urban (between 0 and 100)}
#' }
#' @param areapa A list with variables:
#' \describe{
#'   \item{area}{name of area}
#'   \item{spatialArea}{spatial area of the subarea (e.g. in km^2)}
#' }
#' @param areapsub A list with variables:
#' \describe{
#'   \item{subarea}{name of subarea}
#'   \item{spatialArea}{spatial area of the subarea (e.g. in km^2)}
#' }
#' @param km.res The resolution of the pixelated grid in km
#' @param domain.map.dat A shapefile representing the full spatial domain (e.g. country)
#' @param east.lim Range in km easting over the spatial domain under the input projection
#' @param north.lim Range in km northing over the spatial domain under the input projection
#' @param map.projection A projection function taking longitude and latitude and returning easting and 
#'                northing in km. Or the inverse if inverse is set to TRUE. For example, 
#'                \code{\link{projKenya}}. Check https://epsg.io/ for example for best projection EPSG codes 
#'                for specific countries
#' @param pop Population density raster
#' @param area.map.dat SpatialPolygonsDataFrame object with area level map information
#' @param subarea.map.dat SpatialPolygonsDataFrame object with subarea level map information
#' @param areaNameVar The name of the area variable associated with \code{area.map.dat@data} 
#'            and \code{subarea.map.dat@data}
#' @param subareaNameVar The name of the subarea variable associated with \code{subarea.map.dat@data}
#' @param stratify.by.urban Whether to stratify the pixellated grid by urban/rural. If TRUE, 
#'   renormalizes population densities within areas or subareas crossed with urban/rural
#' @param poppa.target Target population per area stratified by urban rural. Same format as poppa
#' @param adjust.by Whether to adjust population density by the `area` or `subarea` level
#' @param area.polygon.subset.I Index in area.map.dat for a specific area to subset the grid over. This 
#'            option can help reduce computation time relative to constructing the whole grid 
#'            and subsetting afterwards
#' @param subarea.polygon.subset.I FOR EXPERIMENTAL PURPOSES ONLY. Index in subarea.map.dat for a 
#'            specific area to subset the grid over. This 
#'            option can help reduce computation time relative to constructing the whole grid 
#'            and subsetting afterwards
#' @param custom.subset.polygons 'SpatialPolygonsDataFrame' or 'SpatialPolygons' object to subset 
#'            the grid over. This option can help reduce computation time relative to 
#'            constructing the whole grid and subsetting afterwards. `area.polygon.subset.I` or 
#'            `subarea.polygon.subset.I` can be used when subsetting by areas or subareas in 
#'            `area.map.dat` or `subarea.map.dat`. Must be in latitude/longitude projection "EPSG:4326"
#' @param return.popp.tables If TRUE, poppa and poppsub will be calculated based on the generated 
#'                          population integration matrix and input area/subarea map data
#' @param mean.neighbor For determining what area or subarea points are nearest to if they do not 
#' directly fall into an area. See \code{\link[fields]{fields.rdist.near}} for details.
#' @param delta For determining what area or subarea points are nearest to if they do not 
#' directly fall into an area. See \code{\link[fields]{fields.rdist.near}} for details.
#' @param set.na.to.zero If TRUE, sets NA populations to 0.
#' @param fix.zero.pop.density.subareas If TRUE, if population density in a subarea is estimated to be 
#' zero, but the total population in the subarea is nonzero, population is filled into the 
#' area uniformly
#' @param extract.method Either 'bilinear' or 'simple'. see `method` from 
#' \code{\link[terra]{extract}}
#' 
#' @author John Paige
#' @seealso \code{\link{setThresholdsByRegion}}, \code{\link{poppRegionFromPopMat}}, \code{\link{simPopSPDE}}, \code{\link{simPopCustom}}
#' @examples 
#' \dontrun{
#' 
#' library(sp)
#' library(sf)
#' # download Kenya GADM shapefiles from SUMMERdata github repository
#' githubURL <- paste0("https://github.com/paigejo/SUMMERdata/blob/main/data/", 
#'                     "kenyaMaps.rda?raw=true")
#' tempDirectory = "~/"
#' mapsFilename = paste0(tempDirectory, "/kenyaMaps.rda")
#' if(!file.exists(mapsFilename)) {
#'   download.file(githubURL,mapsFilename)
#' }
#' 
#' # load it in
#' out = load(mapsFilename)
#' out
#' adm1@data$NAME_1 = as.character(adm1@data$NAME_1)
#' adm1@data$NAME_1[adm1@data$NAME_1 == "Trans Nzoia"] = "Trans-Nzoia"
#' adm1@data$NAME_1[adm1@data$NAME_1 == "Elgeyo-Marakwet"] = "Elgeyo Marakwet"
#' adm2@data$NAME_1 = as.character(adm2@data$NAME_1)
#' adm2@data$NAME_1[adm2@data$NAME_1 == "Trans Nzoia"] = "Trans-Nzoia"
#' adm2@data$NAME_1[adm2@data$NAME_1 == "Elgeyo-Marakwet"] = "Elgeyo Marakwet"
#' 
#' # some Admin-2 areas have the same name
#' adm2@data$NAME_2 = as.character(adm2@data$NAME_2)
#' adm2@data$NAME_2[(adm2@data$NAME_1 == "Bungoma") & 
#'                    (adm2@data$NAME_2 == "Lugari")] = "Lugari, Bungoma"
#' adm2@data$NAME_2[(adm2@data$NAME_1 == "Kakamega") & 
#'                    (adm2@data$NAME_2 == "Lugari")] = "Lugari, Kakamega"
#' adm2@data$NAME_2[(adm2@data$NAME_1 == "Meru") & 
#'                    (adm2@data$NAME_2 == "Igembe South")] = "Igembe South, Meru"
#' adm2@data$NAME_2[(adm2@data$NAME_1 == "Tharaka-Nithi") & 
#'                    (adm2@data$NAME_2 == "Igembe South")] = "Igembe South, Tharaka-Nithi"
#' 
#' # The spatial area of unknown 8 is so small, it causes problems unless its removed or 
#' # unioned with another subarea. Union it with neighboring Kakeguria:
#' newadm2 = adm2
#' unknown8I = which(newadm2$NAME_2 == "unknown 8")
#' newadm2$NAME_2[newadm2$NAME_2 %in% c("unknown 8", "Kapenguria")] <- 
#'   "Kapenguria + unknown 8"
#' admin2.IDs <- newadm2$NAME_2
#' 
#' newadm2@data = cbind(newadm2@data, NAME_2OLD = newadm2@data$NAME_2)
#' newadm2@data$NAME_2OLD = newadm2@data$NAME_2
#' newadm2@data$NAME_2 = admin2.IDs
#' newadm2$NAME_2 = admin2.IDs
#' temp <- terra::aggregate(as(newadm2, "SpatVector"), by="NAME_2")
#' 
#' temp <- sf::st_as_sf(temp)
#' temp <- sf::as_Spatial(temp)
#' 
#' tempData = newadm2@data[-unknown8I,]
#' tempData = tempData[order(tempData$NAME_2),]
#' newadm2 <- sp::SpatialPolygonsDataFrame(temp, tempData, match.ID = F)
#' adm2 = newadm2
#' 
#' # download 2014 Kenya population density TIF file
#' 
#' githubURL <- paste0("https://github.com/paigejo/SUMMERdata/blob/main/data/", 
#'                     "Kenya2014Pop/worldpop_total_1y_2014_00_00.tif?raw=true")
#' popTIFFilename = paste0(tempDirectory, "/worldpop_total_1y_2014_00_00.tif")
#' if(!file.exists(popTIFFilename)) {
#'   download.file(githubURL,popTIFFilename)
#' }
#' 
#' # load it in
#' pop = terra::rast(popTIFFilename)
#' 
#' east.lim = c(-110.6405, 832.4544)
#' north.lim = c(-555.1739, 608.7130)
#' 
#' ## Construct poppsubKenya, a table of urban/rural general population totals 
#' ## in each subarea. Technically, this is not necessary since we can load in 
#' ## poppsubKenya via data(kenyaPopulationData). First, we will need to calculate 
#' ## the areas in km^2 of the areas and subareas
#' 
#' 
#' # use Lambert equal area projection of areas (Admin-1) and subareas (Admin-2)
#' midLon = mean(adm1@bbox[1,])
#' midLat = mean(adm1@bbox[2,])
#' p4s = paste0("+proj=laea +x_0=0 +y_0=0 +lon_0=", midLon, 
#'              " +lat_0=", midLat, " +units=km")
#' 
#' adm1_sf = st_as_sf(adm1)
#' adm1proj_sf = st_transform(adm1_sf, p4s)
#' adm1proj = as(adm1proj_sf, "Spatial")
#' 
#' adm2_sf = st_as_sf(adm2)
#' adm2proj_sf = st_transform(adm2_sf, p4s)
#' adm2proj = as(adm2proj_sf, "Spatial")
#' 
#' # now calculate spatial area in km^2
#' admin1Areas = as.numeric(st_area(adm1proj_sf))
#' admin2Areas = as.numeric(st_area(adm2proj_sf))
#' 
#' areapaKenya = data.frame(area=adm1proj@data$NAME_1, spatialArea=admin1Areas)
#' areapsubKenya = data.frame(area=adm2proj@data$NAME_1, subarea=adm2proj@data$NAME_2, 
#'                            spatialArea=admin2Areas)
#' 
#' # Calculate general population totals at the subarea (Admin-2) x urban/rural 
#' # level and using 1km resolution population grid. Assign urbanicity by 
#' # thresholding population density based on estimated proportion population 
#' # urban/rural, making sure total area (Admin-1) urban/rural populations in 
#' # each area matches poppaKenya.
#' require(fields)
#' # NOTE: the following function will typically take ~15-20 minutes. Can speed up 
#' #       by setting km.res to be higher, but we recommend fine resolution for 
#' #       this step, since it only needs to be done once. Instead of running 
#' #       the code in the following if(FALSE) section, 
#' #       you can simply run data(kenyaPopulationData)
#' if(FALSE){
#'   system.time(poppsubKenya <- getPoppsub(
#'     km.res=1, pop=pop, domain.map.dat=adm0,
#'     east.lim=east.lim, north.lim=north.lim, map.projection=projKenya,
#'     poppa = poppaKenya, areapa=areapaKenya, areapsub=areapsubKenya, 
#'     area.map.dat=adm1, subarea.map.dat=adm2, 
#'     areaNameVar = "NAME_1", subareaNameVar="NAME_2"))
#' }
#' data(kenyaPopulationData)
#' 
#' # Now generate a general population integration table at 5km resolution, 
#' # based on subarea (Admin-2) x urban/rural population totals. This takes 
#' # ~1 minute
#' system.time(pop.matKenya <- makePopIntegrationTab(
#'   km.res=5, pop=pop, domain.map.dat=adm0,
#'   east.lim=east.lim, north.lim=north.lim, map.projection=projKenya,
#'   poppa = poppaKenya, poppsub=poppsubKenya, 
#'   area.map.dat = adm1, subarea.map.dat = adm2,
#'   areaNameVar = "NAME_1", subareaNameVar="NAME_2"))
#' 
#' ## Adjust pop.mat to be target (neonatal) rather than general population density. First
#' ## create the target population frame
#' ## (these numbers are based on IPUMS microcensus data)
#' mothersPerHouseholdUrb = 0.3497151
#' childrenPerMotherUrb = 1.295917
#' mothersPerHouseholdRur = 0.4787696
#' childrenPerMotherRur = 1.455222
#' targetPopPerStratumUrban = easpaKenya$HHUrb * mothersPerHouseholdUrb * childrenPerMotherUrb
#' targetPopPerStratumRural = easpaKenya$HHRur * mothersPerHouseholdRur * childrenPerMotherRur
#' easpaKenyaNeonatal = easpaKenya
#' easpaKenyaNeonatal$popUrb = targetPopPerStratumUrban
#' easpaKenyaNeonatal$popRur = targetPopPerStratumRural
#' easpaKenyaNeonatal$popTotal = easpaKenyaNeonatal$popUrb + easpaKenyaNeonatal$popRur
#' easpaKenyaNeonatal$pctUrb = 100 * easpaKenyaNeonatal$popUrb / easpaKenyaNeonatal$popTotal
#' easpaKenyaNeonatal$pctTotal = 
#'   100 * easpaKenyaNeonatal$popTotal / sum(easpaKenyaNeonatal$popTotal)
#' 
#' # Generate the target population density by scaling the current population density grid 
#' # at the Admin1 x urban/rural level
#' pop.matKenyaNeonatal = adjustPopMat(pop.matKenya, easpaKenyaNeonatal)
#' 
#' # Generate neonatal population table from the neonatal population integration matrix.
#' # This is technically not necessary for population simulation purposes, but is here 
#' # for illustrative purposes
#' poppsubKenyaNeonatal = poppRegionFromPopMat(pop.matKenyaNeonatal, pop.matKenyaNeonatal$subarea)
#' poppsubKenyaNeonatal = cbind(subarea=poppsubKenyaNeonatal$region, 
#'                              area=adm2@data$NAME_1[match(poppsubKenyaNeonatal$region, 
#'                                adm2@data$NAME_2)], 
#'                              poppsubKenyaNeonatal[,-1])
#' print(head(poppsubKenyaNeonatal))
#' }
#' @importFrom terra extract
#' @importFrom terra gdal
#' @importFrom terra crs<-
#' @importFrom sp SpatialPoints
#' @importFrom sp CRS
#' @importFrom sp over
#' @importFrom sp coordinates
#' @importFrom sp as.SpatialPolygons.PolygonsList
#' @importFrom fields make.surface.grid
#' @export
makePopIntegrationTab = function(km.res=5, pop, domain.map.dat, east.lim, north.lim, map.projection, 
                                 area.map.dat, subarea.map.dat, 
                                 areaNameVar="NAME_1", subareaNameVar="NAME_2", 
                                 poppa=NULL, poppsub=NULL, stratify.by.urban=TRUE, 
                                 areapa=NULL, areapsub=NULL, custom.subset.polygons=NULL, 
                                 area.polygon.subset.I=NULL, subarea.polygon.subset.I=NULL, 
                                 mean.neighbor=50, delta=.1, return.popp.tables=FALSE, 
                                 set.na.to.zero=TRUE, fix.zero.pop.density.subareas=FALSE, 
                                 extract.method="bilinear") {
  thresholdUrbanBy = ifelse(is.null(poppsub), "area", "subarea")
  
  # some basic checks to make sure inputs are correct
  
  # make sure area and subarea names are unique
  nameCounts = aggregate(area.map.dat@data[[areaNameVar]], by=list(area=area.map.dat@data[[areaNameVar]]), FUN=length)
  if(any(nameCounts$x > 1)) {
    stop("area names are not unique in area.map.dat")
  }
  if(!is.null(subarea.map.dat)) {
    nameCounts = aggregate(subarea.map.dat@data[[subareaNameVar]], by=list(area=subarea.map.dat@data[[subareaNameVar]]), FUN=length)
    if(any(nameCounts$x > 1)) {
      stop("subarea names are not unique in subarea.map.dat")
    }
  }
  if(!is.null(poppa)) {
    nameCounts = aggregate(poppa$area, by=list(area=poppa$area), FUN=length)
    if(any(nameCounts$x > 1)) {
      stop("area names are not unique in poppa")
    }
    
    # make sure area names match up
    if(!all.equal(sort(poppa$area), sort(area.map.dat@data[[areaNameVar]]))) {
      stop("area names in poppa do not match those in area.map.dat@data")
    }
  }
  if(!is.null(poppsub)) {
    nameCounts = aggregate(poppsub$subarea, by=list(area=poppsub$subarea), FUN=length)
    if(any(nameCounts$x > 1)) {
      stop("subarea names are not unique in poppsub")
    }
    
    # make sure subarea names match up
    if(!is.null(subarea.map.dat)) {
      if(!all.equal(sort(poppsub$subarea), sort(subarea.map.dat@data[[subareaNameVar]]))) {
        stop("subarea names in poppsub do not match those in subarea.map.dat@data")
      }
    }
  }
  
  # get a rectangular grid
  eastGrid = seq(east.lim[1], east.lim[2], by=km.res)
  northGrid = seq(north.lim[1], north.lim[2], by=km.res)
  
  if(!is.null(area.polygon.subset.I)) {
    # get range of the grid that we actually need
    temp = area.map.dat@polygons[[area.polygon.subset.I]]
    allPolygons = temp@Polygons
    eastNorthCoords = do.call("rbind", lapply(1:length(allPolygons), function(i) {map.projection(allPolygons[[i]]@coords)}))
    eastSubRange = range(eastNorthCoords[,1])
    northSubRange = range(eastNorthCoords[,2])
    
    # subset grid to the range we need
    eastGrid = eastGrid[eastGrid >= eastSubRange[1]]
    eastGrid = eastGrid[eastGrid <= eastSubRange[2]]
    northGrid = northGrid[northGrid >= northSubRange[1]]
    northGrid = northGrid[northGrid <= northSubRange[2]]
  }
  
  if(!is.null(subarea.polygon.subset.I)) {
    # get range of the grid that we actually need
    temp = subarea.map.dat@polygons[[subarea.polygon.subset.I]]
    allPolygons = temp@Polygons
    eastNorthCoords = do.call("rbind", lapply(1:length(allPolygons), function(i) {map.projection(allPolygons[[i]]@coords)}))
    eastSubRange = range(eastNorthCoords[,1])
    northSubRange = range(eastNorthCoords[,2])
    
    # subset grid to the range we need
    eastGrid = eastGrid[eastGrid >= eastSubRange[1]]
    eastGrid = eastGrid[eastGrid <= eastSubRange[2]]
    northGrid = northGrid[northGrid >= northSubRange[1]]
    northGrid = northGrid[northGrid <= northSubRange[2]]
  }
  
  if(!is.null(custom.subset.polygons)) {
    if(("SpatialPolygons" %in% class(custom.subset.polygons)) || ("SpatialPolygonsDataFrame" %in% class(custom.subset.polygons))) {
      
      # calculate the east/north range of each polygon
      getPolygonENrange = function(pol) {
        # get range of the grid that we actually need
        temp = pol
        allPolygons = temp@Polygons
        eastNorthCoords = do.call("rbind", lapply(1:length(allPolygons), function(i) {map.projection(allPolygons[[i]]@coords)}))
        eastSubRange = range(eastNorthCoords[,1])
        northSubRange = range(eastNorthCoords[,2])
        
        # subset grid to the range we need
        rbind(eastSubRange, 
              northSubRange)
      }
      allRanges = lapply(getPolygonENrange, custom.subset.polygons@polygons)
      
      # calculate the full range based on each individual polygon range
      eastings = sapply(allRanges, function(x) {x[1,]})
      northings = sapply(allRanges, function(x) {x[2,]})
      eastSubRange = range(eastings)
      northSubRange = range(northings)
      
      # subset grid to the range we need
      eastGrid = eastGrid[eastGrid >= eastSubRange[1]]
      eastGrid = eastGrid[eastGrid <= eastSubRange[2]]
      northGrid = northGrid[northGrid >= northSubRange[1]]
      northGrid = northGrid[northGrid <= northSubRange[2]]
    } else {
      stop("custom.subset.polygons must be of class 'SpatialPolygons' or 'SpatialPolygonsDataFrame'")
    }
  }
  
  utmGrid = matrix(fields::make.surface.grid(list(east=eastGrid, north=northGrid)), ncol=2)
  
  # project coordinates into lat/lon
  if(length(utmGrid) > 0) {
    lonLatGrid = matrix(map.projection(utmGrid, inverse=TRUE), ncol=2)
  } else {
    warning(paste0("no grid cell centroid are in the areas of interest. ", 
                   "Integration grid will be composed entirely of custom ", 
                   "integration points at the centroids of the areas of interest"))
    lonLatGrid = utmGrid
  }
  
  # subset grid so it's in the domain
  # inDomain = in.poly(lonLatGrid, domainPoly)
  # determine version of PROJ
  ver = terra::gdal(lib="proj")
  PROJ6 <- as.numeric(substr(ver, 1, 1)) >= 6
  
  # from lon/lat coords to easting/northing
  # if(!PROJ6) {
  #   lonLatCoords = sp::SpatialPoints(lonLatGrid, proj4string=sp::CRS("+proj=longlat"))
  # } else {
  #   lonLatCoords = sp::SpatialPoints(lonLatGrid, proj4string=sp::CRS(SRS_string="EPSG:4326"))
  # }
  lonLatCoords = sp::SpatialPoints(lonLatGrid, proj4string=domain.map.dat@proj4string)
  inDomain = sp::over(lonLatCoords, domain.map.dat)
  inDomain = !is.na(inDomain[,1])
  utmGrid = matrix(utmGrid[inDomain,], ncol=2)
  lonLatGrid = matrix(lonLatGrid[inDomain,], ncol=2)
  
  # compute areas associated with locations
  if(length(lonLatGrid) > 0) {
    if(!is.null(subarea.map.dat)) {
      subareas = SUMMER::getAreaName(lonLatGrid, subarea.map.dat, areaNameVar=subareaNameVar, mean.neighbor=mean.neighbor, delta=delta)$areaNames
    } else {
      subareas = NULL
    }
    
    if(!is.null(area.map.dat)) {
      areas = SUMMER::getAreaName(lonLatGrid, area.map.dat, areaNameVar=areaNameVar, mean.neighbor=mean.neighbor, delta=delta)$areaNames
    } else {
      areas = NULL
    }
    
  } else {
    subareas = character(0)
    areas = character(0)
  }
  
  if(!is.null(area.polygon.subset.I)) {
    areaSubsetName = area.map.dat@data[area.polygon.subset.I,areaNameVar]
    insideArea = areas == areaSubsetName
    
    # subset grid and area/subarea names so they're in the area of interest
    utmGrid = matrix(utmGrid[insideArea,], ncol=2)
    lonLatGrid = matrix(lonLatGrid[insideArea,], ncol=2)
    areas = areas[insideArea]
    subareas = subareas[insideArea]
  }
  
  if(!is.null(subarea.polygon.subset.I)) {
    subareaSubsetName = subarea.map.dat@data[subarea.polygon.subset.I,subareaNameVar]
    insideSubarea = subareas == subareaSubsetName
    
    # subset grid and area/subarea names so they're in the area of interest
    utmGrid = matrix(utmGrid[insideSubarea,], ncol=2)
    lonLatGrid = matrix(lonLatGrid[insideSubarea,], ncol=2)
    areas = areas[insideSubarea]
    subareas = subareas[insideSubarea]
  }
  
  if(!is.null(custom.subset.polygons)) {
    if(!grepl("+proj=longlat", custom.subset.polygons@proj4string@projargs)) {
      warning(paste0("custom.subset.polygons does not use longitude/latitude ", 
                     "coordinate system? Using makePopIntegrationTab outside ", 
                     "its use case, which may result in errors"))
    }
    
    # if(!PROJ6) {
    #   lonLatCoords = sp::SpatialPoints(lonLatGrid, proj4string=sp::CRS("+proj=longlat"))
    # } else {
    #   lonLatCoords = sp::SpatialPoints(lonLatGrid, proj4string=sp::CRS(SRS_string="EPSG:4326"))
    # }
    lonLatCoords = sp::SpatialPoints(lonLatGrid, proj4string=custom.subset.polygons@proj4string)
    insideCustomSubset = sp::over(lonLatCoords, custom.subset.polygons)
    
    # subset grid and area/subarea names so they're in the area of interest
    utmGrid = matrix(utmGrid[insideCustomSubset,], ncol=2)
    lonLatGrid = matrix(lonLatGrid[insideCustomSubset,], ncol=2)
    areas = areas[insideCustomSubset]
    
    if(!is.null(subarea.map.dat)) {
      subareas = subareas[insideCustomSubset]
    }
  }
  
  # determine what subareas we want our grid to contain
  if(!is.null(subarea.map.dat)) {
    allSubareas = sort(unique(subarea.map.dat@data[[subareaNameVar]]))
    if(!is.null(area.polygon.subset.I)) {
      allSubareas = sort(unique(subarea.map.dat@data[subarea.map.dat@data[[areaNameVar]] == areaSubsetName,][[subareaNameVar]]))
    }
    if(!is.null(subarea.polygon.subset.I)) {
      allSubareas = subareaSubsetName
    }
    if(!is.null(custom.subset.polygons)) {
      allSubareas = sort(unique(subareas))
    }
  }
  
  # filter out parts of poppsub that are irrelevant for the subareas of 
  # interest
  if(!is.null(poppsub)) {
    poppsub = poppsub[poppsub$subarea %in% allSubareas,]
  }
  
  if(!is.null(subarea.map.dat)) {
    # check to make sure every subarea has at least 2 pixels
    subareasFactor = factor(subareas, levels=allSubareas)
    if(length(lonLatGrid) > 0) {
      out = aggregate(subareas, by=list(subarea=subareasFactor), FUN=length, drop=FALSE)
    } else {
      out = data.frame(subarea=sort(unique(subarea.map.dat@data[[subareaNameVar]])), 
                       x=NA)
    }
    noPixels = is.na(out$x)
    onePixel = out$x == 1
    onePixel[is.na(onePixel)] = FALSE
    onePixelNames = out$subarea[onePixel]
    badSubareas = noPixels | onePixel
    badSubareaNames = as.character(out$subarea[badSubareas])
    
    if(any(badSubareas)) {
      badSubareaString = paste(badSubareaNames, collapse=", ")
      warning(paste0("The following subareas have < 2 regular grid points ", 
                     "at the given resolution: ", badSubareaString, ", so ", 
                     "they will be given custom integration points"))
      
      # get centroids of the subareas (or it's single pixel coordinates)
      thisSpatialPolyList = sp::as.SpatialPolygons.PolygonsList(subarea.map.dat@polygons)
      centroidsLonLat = matrix(ncol=2, nrow=length(allSubareas))
      
      for(i in 1:length(allSubareas)) {
        thisSubarea = allSubareas[i]
        if(thisSubarea %in% onePixelNames) {
          thisCentroid = lonLatGrid[subareas == thisSubarea,]
        } else {
          subareaI = match(as.character(thisSubarea), as.character(subarea.map.dat@data[[subareaNameVar]]))
          thisCentroid = sp::coordinates(thisSpatialPolyList[subareaI])
        }
        
        centroidsLonLat[i,] = thisCentroid
      }
      
      # sort to match results of aggregate (alphabetical order)
      # sortI = sort(as.character(subarea.map.dat@data[[subareaNameVar]]), index.return=TRUE)$ix
      sortI = sort(as.character(allSubareas), index.return=TRUE)$ix
      centroidsLonLat = centroidsLonLat[sortI,]
      
      # remove the one pixel for subareas with only one pixel 
      # (we will add it in again later, twice if stratified: both urban and rural)
      onePixel = which(subareas %in% onePixelNames)
      if(length(onePixel) > 0) {
        lonLatGrid = lonLatGrid[-onePixel,]
        utmGrid = utmGrid[-onePixel,]
        subareas = subareas[-onePixel]
        areas = areas[-onePixel]
      }
      
      # add centroids of only the bad subareas
      centroidsLonLat = matrix(centroidsLonLat[badSubareas,], ncol=2)
      
      centroidsEastNorth = map.projection(centroidsLonLat[,1], centroidsLonLat[,2])

      # only add centroid in stratum if bad subareas have any population in the stratum. 
      # If poppsub not included and resolution not high enough to have multiple points 
      # in a subarea, treat it as entirely urban or rural
      if(is.null(poppsub)) {
        hasUrbanPop = rep(TRUE, sum(badSubareas))
        hasRuralPop = rep(FALSE, sum(badSubareas))
      } else {
        hasUrbanPop = (poppsub$popUrb > 0)[badSubareas]
        hasRuralPop = (poppsub$popRur > 0)[badSubareas]
      }
      
      # add centroids to the matrices of pixellated grid coordinates. 
      # Add them twice: once for urban, once for rural
      lonLatGrid = rbind(lonLatGrid, centroidsLonLat[hasUrbanPop,], centroidsLonLat[hasRuralPop,])
      utmGrid = rbind(utmGrid, centroidsEastNorth[hasUrbanPop,], centroidsEastNorth[hasRuralPop,])
      
      # add associated consituencies, areas, provinces to respective vectors
      # Normally, we could just caluclate what subarea/area the points are in. 
      # However, since centroids might not be in the associated subarea, we 
      # instead just assign the known subareas directly
      # newSubareas = getAreaName(rbind(centroidsLonLat[hasUrbanPop,], 
      #                                 centroidsLonLat[hasRuralPop,]), 
      #                           subarea.map.dat, subareaNameVar, delta, mean.neighbor)$areaNames
      # newAreas = getAreaName(rbind(centroidsLonLat[hasUrbanPop,], 
      #                              centroidsLonLat[hasRuralPop,]), 
      #                        area.map.dat, areaNameVar, delta, mean.neighbor)$areaNames
      newSubareas = c(badSubareaNames[hasUrbanPop], badSubareaNames[hasRuralPop])
      newAreas = subarea.map.dat@data[[areaNameVar]][match(newSubareas, subarea.map.dat@data[[subareaNameVar]])]
      
      subareas = c(as.character(subareas), as.character(newSubareas))
      areas = c(as.character(areas), as.character(newAreas))
    }
  } else {
    badSubareas = FALSE
  }
  
  # get population density at those coordinates
  if(!PROJ6) {
    # extract the raster values for each chunk of points
    interpPopVals <- tryCatch(
      {
        terra::extract(pop, lonLatGrid,method=extract.method)
      },
      error=function(cond) {
        message(cond)
        stop(paste0("Error extracting raster values. In case of memory limitations, see ", 
                    "terra::terraOptions()"))
        # Choose a return value in case of error
        return(NA)
      }
    )
  } else {
    # make sure CRS string of population density SpatRaster is "EPSG:4326", i.e. 
    # longitude + latitude
    tempCRSstr = sp::CRS(SRS_string="EPSG:4326")
    crsStr = attr(tempCRSstr, "comment")
    if(is.null(crsStr)) {
      crsStr <- tempCRSstr@projargs
      if(is.na(crsStr)) {
        crsStr <- ""
      }
    }
    terra::crs(pop) = crsStr
    
    # get population density values from the raster
    interpPopVals <- tryCatch(
      {
        terra::extract(pop, lonLatGrid, method="bilinear")
      },
      error=function(cond) {
        message(cond)
        stop(paste0("Error extracting raster values. In case of memory limitations, see ", 
                    "terra::aggregate, terra::resample, terra::projectRaster, and terra::terraOptions"))
        # Choose a return value in case of error
        return(NA)
      }
    )
  }
  interpPopVals = unlist(interpPopVals)
  names(interpPopVals) = NULL
  
  if(set.na.to.zero) {
    interpPopVals[is.na(interpPopVals)] = 0
  }
  
  # if requested, fix subareas with entirely zero population density
  if(fix.zero.pop.density.subareas && !is.null(poppsub)) {
    newPop = data.frame(list(lon=lonLatGrid[,1], lat=lonLatGrid[,2], pop=interpPopVals, area=areas, subarea=subareas, urban=NA))
    poppsubCurrent = poppRegionFromPopMat(newPop, newPop$subarea)
    
    zeroPopSubareas = poppsubCurrent$region[poppsubCurrent$popTotal == 0]
    if(length(zeroPopSubareas) > 0) {
      warning(paste0("The following subareas have entirely zero population density ", 
                     "but nonzero total population, and their population will be filled ", 
                     "in uniformly: ", paste(zeroPopSubareas, collapse=", ")))
      # fill in population density uniformly in these subareas
      for(i in 1:length(zeroPopSubareas)) {
        thisSub = zeroPopSubareas[i]
        thisSubI = newPop$subarea == thisSub
        thisSubPop = poppsubCurrent$popTotal[poppsubCurrent$region == thisSub]
        newPop$pop[thisSubI] = thisSubPop/sum(thisSubI)
      }
    }
    
    interpPop = newPop$pop
  }
  
  if(any(badSubareas)) {
    # make sure population densities in the bad subareas 
    # are slightly different so one will be classified as urban and 
    # the other as rural. They will be renormalized later based on poppsub, 
    # or, if poppsub doesn't exist, only new/custom "urban" points will be kept, 
    # so no densities are modified here
    nUnits = length(interpPopVals)
    nNewRural = sum(hasRuralPop)
    if(nNewRural >= 1) {
      interpPopVals[(nUnits-nNewRural + 1):nUnits] = interpPopVals[(nUnits-nNewRural + 1):nUnits] / 2
    }
  }
  
  # determine which points are urban via population density thresholding
  newPop = data.frame(list(lon=lonLatGrid[,1], lat=lonLatGrid[,2], pop=interpPopVals, area=areas, subarea=subareas, urban=NA))
  if(thresholdUrbanBy == "area") {
    threshes = SUMMER::setThresholdsByRegion(newPop, poppa, region.type="area")
    popThreshes = sapply(1:nrow(newPop), function(i) {threshes$threshes[as.character(threshes$regions) == as.character(newPop$area[i])]})
    urban = newPop$pop >= unlist(popThreshes)
    newPop$urban = urban
  } else {
    tempSubarea = poppsub
    tempPop = newPop
    allSubareas = sort(unique(tempPop$subarea))
    tempSubarea = tempSubarea[tempSubarea$subarea %in% allSubareas,]
    threshes = SUMMER::setThresholdsByRegion(tempPop, poppr = tempSubarea, region.type="subarea")
    popThreshes = sapply(1:nrow(newPop), function(i) {threshes$threshes[as.character(threshes$regions) == as.character(newPop$subarea[i])]})
    urban = newPop$pop >= unlist(popThreshes)
    newPop$urban = urban
  }
  
  newPop$east = utmGrid[,1]
  newPop$north = utmGrid[,2]
  
  pointAreaNames = as.character(newPop$area)
  pointAreaNamesU = as.character(newPop$area)
  pointAreaNamesU[!newPop$urban] = "DoNotUseThis"
  pointAreaNamesR = as.character(newPop$area)
  pointAreaNamesR[newPop$urban] = "DoNotUseThis"
  
  if(!is.null(subarea.map.dat)) {
    pointSubareaNames = as.character(newPop$subarea)
    pointSubareaNamesU = as.character(newPop$subarea)
    pointSubareaNamesU[!newPop$urban] = "DoNotUseThis"
    pointSubareaNamesR = as.character(newPop$subarea)
    pointSubareaNamesR[newPop$urban] = "DoNotUseThis"
  }
  
  # if necessary, renormalize population values within subareas or areas 
  # crossed with urban/rural to be the correct value
  if(!is.null(poppsub)) {
    # In this case, areapsub and areapa are not necessary
    
    if(stratify.by.urban) {
      pointPopUrb = calibrateByRegion(newPop$pop, pointSubareaNamesU, poppsub$subarea, poppsub$popUrb)
      pointPopRur = calibrateByRegion(newPop$pop, pointSubareaNamesR, poppsub$subarea, poppsub$popRur)
      newPop$pop = (pointPopUrb + pointPopRur)
    } else {
      newPop$pop = calibrateByRegion(newPop$pop, pointSubareaNames, poppsub$subarea, poppsub$popTotal)
    }
  } else if(!is.null(poppa)) {
    # i.e. we don't have poppsub, so now we need to be careful about the spatial areas 
    # (e.g. area in km^2) associated with each point
    
    # determine how to use spatial areas (e.g. in km^2) of areas/subareas:
    # Case 1: We don't have subarea.map.dat, but do have areapa
    # Case 2: We don't have subarea.map.dat, and don't have areapa
    # Case 3: We have subarea.map.dat and areapsub
    # Case 4: We have subarea.map.dat but don't have areapsub (this is sketchy if 
    #         there's any custom integration points)
    
    # set the areas of each point to be equal to start out. This must be 
    # proportional to, but not necessarily equal to, the actual area 
    # associated with each point
    pointAreas = rep(1, nrow(newPop))
    if(is.null(subarea.map.dat) && !is.null(areapa)) {
      pointAreas = calibrateByRegion(pointAreas, pointAreaNames, areapa$area, areapa$spatialArea)
    } else if(is.null(subarea.map.dat) && is.null(areapa)) {
      # do nothing, since pointAreas = rep(1, nrow(newPop)) is fine
    } else if(!is.null(subarea.map.dat) && !is.null(areapsub)) {
      pointAreas = calibrateByRegion(pointAreas, pointSubareaNames, areapsub$subarea, areapsub$spatialArea)
    } else if(!is.null(subarea.map.dat) && is.null(areapsub)) {
      if(any(badSubareas)) {
        stop("poppsub and areapsub are NULL, but subarea.map.dat is included")
      }
      # could continue here, since pointAreas = rep(1, nrow(newPop)) might not be disastrous, 
      # but it would be wrong
    }
    
    # use the population per stratum to renormalize population densities
    if(stratify.by.urban) {
      popUrb = calibrateByRegion(newPop$pop*pointAreas, pointAreaNamesU, poppa$area, poppa$popUrb)
      popRur = calibrateByRegion(newPop$pop*pointAreas, pointAreaNamesR, poppa$area, poppa$popRur)
      
      newPop$pop = popUrb + popRur
    } else {
      newPop$pop = calibrateByRegion(newPop$pop*pointAreas, pointAreaNames, poppa$area, poppa$popTotal)
    }
  } else {
    stop("must either include poppsub, or poppa")
  }
  
  # if requested by user, return the population integration matrix as well 
  # as the associate population per area and subarea tables
  if(return.popp.tables) {
    if(!is.null(area.map.dat)) {
      areas = getAreaName(cbind(newPop$lon, newPop$lat), shapefile=area.map.dat, areaNameVar=areaNameVar)$areaNames
      newpoppa = poppRegionFromPopMat(newPop, areas)
      names(newpoppa)[1] = "area"
      newpoppa$pctTotal = 100 * newpoppa$popTotal/sum(newpoppa$popTotal)
      newpoppa$pctUrb = 100 * newpoppa$popUrb/newpoppa$popTotal
    } else {
      if(is.null(poppa)) {
        warning(paste0("return.popp.tables set to TRUE, but area.map.dat is NULL, ", 
                       "so no area level population table will be calculated"))
      }
      newpoppa = NULL
    }
    if(!is.null(subarea.map.dat)) {
      subareas = getAreaName(cbind(newPop$lon, newPop$lat), shapefile=subarea.map.dat, areaNameVar=subareaNameVar)$areaNames
      newpoppsub = poppRegionFromPopMat(newPop, subareas)
      names(newpoppsub)[1] = "subarea"
      
      # get area associated with subareas
      associatedAreas = sapply(newpoppsub$subarea, function(subareaName) {newPop$area[match(subareaName, newPop$subarea)]})
      newpoppsub$area = associatedAreas
      
      newpoppsub = newpoppsub[,c("subarea", "area", "popUrb", "popRur", "popTotal")]
      
      # calculate percent urban and percent of total population
      newpoppsub$pctUrb = 100 * newpoppsub$popUrb / newpoppsub$popTotal
      newpoppsub$pctTotal = 100 * newpoppsub$popTotal / sum(newpoppsub$popTotal)
    } else {
      if(is.null(poppsub)) {
        warning(paste0("return.popp.tables set to TRUE, but subarea.map.dat is NULL, ", 
                       "so no subarea level population table will be calculated"))
      }
      newpoppsub = NULL
    }
    
    return(list(pop.mat=newPop, poppa=newpoppa, poppsub=newpoppsub))
  }
  
  newPop
}

#' 
#' `r lifecycle::badge("experimental")`
#' 
#' @describeIn makePopIntegrationTab Generate table of estimates of population
#'   totals per subarea x urban/rural combination based on population density
#'   raster at `kmres` resolution "grid", including custom integration points
#'   for any subarea too small to include grid points at their centroids.
#' @export
getPoppsub = function(km.res=1, pop, domain.map.dat, east.lim, north.lim, map.projection, 
                      poppa, areapa=NULL, areapsub, subarea.map.dat, subareaNameVar="NAME_2", 
                      stratify.by.urban=TRUE, area.map.dat=NULL, areaNameVar="NAME_1", 
                      area.polygon.subset.I=NULL, subarea.polygon.subset.I=NULL, 
                      custom.subset.polygons=NULL, 
                      mean.neighbor=50, delta=.1, set.na.to.zero=TRUE, fix.zero.pop.density.subareas=FALSE) {
  
  out = SUMMER::makePopIntegrationTab(km.res=km.res, pop=pop, domain.map.dat=domain.map.dat, 
                              areapa=areapa, areapsub=areapsub, 
                              east.lim=east.lim, north.lim=north.lim, map.projection=map.projection, 
                              subarea.map.dat=subarea.map.dat, area.map.dat=area.map.dat, 
                              areaNameVar=areaNameVar, subareaNameVar=subareaNameVar, 
                              poppa=poppa, stratify.by.urban=stratify.by.urban, 
                              area.polygon.subset.I=area.polygon.subset.I, 
                              subarea.polygon.subset.I=subarea.polygon.subset.I, 
                              custom.subset.polygons=custom.subset.polygons, 
                              mean.neighbor=mean.neighbor, delta=delta, 
                              set.na.to.zero=set.na.to.zero, 
                              fix.zero.pop.density.subareas=fix.zero.pop.density.subareas, 
                              return.popp.tables=TRUE)
  out$poppsub
}

#' 
#' `r lifecycle::badge("experimental")`
#' 
#' @describeIn makePopIntegrationTab Adjust population densities in grid based on a population frame.
#' @export
adjustPopMat = function(pop.mat, poppa.target=NULL, adjust.by=c("area", "subarea"), stratify.by.urban=TRUE) {
  adjust.by = match.arg(adjust.by)
  
  # sort get population per stratum from poppa.target
  if(adjust.by == "area") {
    areas=sort(unique(poppa.target$area))
  } else {
    areas=sort(unique(poppa.target$subarea))
  }
  
  if(stratify.by.urban) {
    targetPopPerStratumUrban = poppa.target$popUrb
    targetPopPerStratumRural = poppa.target$popRur
    
    # generate 2 nArea x nPixels matrices for urban and rural strata integrating pixels with respect to population density to get area estimates
    getAreaStratumIntegrationMatrix = function(getUrban=TRUE) {
      areas = as.character(areas)
      
      if(adjust.by == "area") {
        mat = t(sapply(areas, function(area) {
          pop.mat$area == area
        }))
      } else {
        mat = t(sapply(areas, function(area) {
          pop.mat$subarea == area
        }))
      }
      
      mat = sweep(mat, 2, pop.mat$pop, "*")
      
      sweep(mat, 2, pop.mat$urban == getUrban, "*")
    }
    urbanIntegrationMat = getAreaStratumIntegrationMatrix()
    ruralIntegrationMat = getAreaStratumIntegrationMatrix(FALSE)
    
    # calculate number of people per stratum by integrating the population density surface
    urbanPopulations = rowSums(urbanIntegrationMat)
    ruralPopulations = rowSums(ruralIntegrationMat)
    
    # adjust each row of the integration matrices to get the correct expected number of children per stratum
    urbanIntegrationMat = sweep(urbanIntegrationMat, 1, targetPopPerStratumUrban / urbanPopulations, "*")
    urbanIntegrationMat[urbanPopulations == 0,] = 0
    ruralIntegrationMat = sweep(ruralIntegrationMat, 1, targetPopPerStratumRural / ruralPopulations, "*")
    ruralIntegrationMat[ruralPopulations == 0,] = 0
    
    # the column sums of the matrices give the correct modified population densities
    pop.mat$pop = colSums(urbanIntegrationMat) + colSums(ruralIntegrationMat)
  } else {
    targetPopPerArea = poppa.target$popTotal
    
    # generate 2 nArea x nPixels matrices for urban and rural strata integrating pixels with respect to population density to get area estimates
    getAreaIntegrationMatrix = function() {
      areas = as.character(areas)
      
      if(adjust.by == "area") {
        mat = t(sapply(areas, function(area) {
          pop.mat$area == area
        }))
      } else {
        mat = t(sapply(areas, function(area) {
          pop.mat$subarea == area
        }))
      }
      
      mat = sweep(mat, 2, pop.mat$pop, "*")
      mat
    }
    integrationMat = getAreaIntegrationMatrix()
    
    # calculate number of people per stratum by integrating the population density surface
    totalPopulations = rowSums(integrationMat)
    
    # adjust each row of the integration matrices to get the correct expected number of children per stratum
    urbanIntegrationMat = sweep(integrationMat, 1, targetPopPerArea / totalPopulations, "*")
    urbanIntegrationMat[totalPopulations == 0,] = 0
    
    # the column sums of the matrices give the correct modified population densities
    pop.mat$pop = colSums(integrationMat)
  }
  
  
  pop.mat
}

#' Calibrate the point level totals so their sum matches the regional totals
#' 
#' Calibrate/normalize the point level totals so their sum matches the 
#' regional totals. Technically, the totals can be at any level smaller  
#' than the region level specified.
#' 
#' 
#' `r lifecycle::badge("experimental")`
#' 
#' @param point.totals Vector of point level totals that will be calibrated/normalized
#' @param point.regions Vector of regions associated with each point
#' @param regions Vector of region names
#' @param region.totals Vector of desired region level totals associated with `regions`
#' 
#' @return A vector of same length as point.totals and point.regions containing 
#' the calibrated/normalized point totals that sum to the correct regional totals
#' 
#' @author John Paige
#' 
#' @examples
#' point.totals = c(1, 1, 1, 2)
#' point.regions = c("a", "a", "b", "b")
#' region.totals = c(10, 20)
#' regions = c("a", "b")
#' calibrateByRegion(point.totals, point.regions, regions, region.totals)
#' 
#' @return Vector of updated point level totals, calibrated to match region totals
#' 
#' @export
calibrateByRegion = function(point.totals, point.regions, regions, region.totals) {
  regions = as.character(regions)
  point.regions = as.character(point.regions)
  
  # generate nRegions x nPoints matrix that separates points totals to each region
  getRegionIntegrationMatrix = function() {
    
    mat = t(sapply(regions, function(regionName) {point.regions == regionName}))
    mat = sweep(mat, 2, point.totals, "*")
    mat
  }
  integrationMat = getRegionIntegrationMatrix()
  
  # calculate total per region before calibration
  tempRegionTotals = rowSums(integrationMat)
  
  # adjust each row of the integration matrices to get the correct totals per region
  integrationMat = sweep(integrationMat, 1, region.totals / tempRegionTotals, "*")
  integrationMat[tempRegionTotals == 0,] = 0
  
  # check to make sure all zero population totals are in zero population regions. If 
  # any extra, spread the region population in the associated point.totals
  badRegions = (tempRegionTotals == 0) & (region.totals != 0)
  if(any(badRegions)) {
    badRegionIs = which(badRegions)
    for(i in 1:length(badRegionIs)) {
      thisRegionI = badRegionIs[i]
      thisRegion = point.regions == regions[thisRegionI]
      integrationMat[thisRegionI,thisRegion] = region.totals[thisRegionI] * (1/sum(thisRegion))
    }
  }
  
  # the column sums of the matrix give the correct calibrated regional totals
  colSums(integrationMat)
}

#' Generate a population frame of a similar format to poppa argument of \code{\link{simPopCustom}} with a custom set of regions
#' 
#' `r lifecycle::badge("experimental")`
#' 
#' 
#' @param pop.mat Pixellated grid data frame with variables `area` and `pop`. Assumed to be stratified by urban/rural
#' @param regions character vector of length nPixels giving a custom set of regions for which to generate 
#'  a population frame using population density
#' 
#' @details Urbanicity thresholds are set based on that region's percent population 
#' urban. Intended as a helper function of \code{\link{getPoppsub}}, but 
#' can also be used for custom sets of regions (i.e. more than just 2 
#' areal levels: area and subarea). 
#' 
#' @author John Paige
#' 
#' @seealso \code{\link{getPoppsub}}
#' 
#' @examples
#' \dontrun{
#' data(kenyaPopulationData)
#' 
#' #' # download Kenya GADM shapefiles from SUMMERdata github repository
#' githubURL <- "https://github.com/paigejo/SUMMERdata/blob/main/data/kenyaMaps.rda?raw=true"
#' tempDirectory = "~/"
#' mapsFilename = paste0(tempDirectory, "/kenyaMaps.rda")
#' if(!file.exists(mapsFilename)) {
#'   download.file(githubURL,mapsFilename)
#' }
#' 
#' # load it in
#' out = load(mapsFilename)
#' out
#' kenyaMesh <- fmesher::fm_as_fm(kenyaMesh)
#' adm1@data$NAME_1 = as.character(adm1@data$NAME_1)
#' adm1@data$NAME_1[adm1@data$NAME_1 == "Trans Nzoia"] = "Trans-Nzoia"
#' adm1@data$NAME_1[adm1@data$NAME_1 == "Elgeyo-Marakwet"] = "Elgeyo Marakwet"
#' adm2@data$NAME_1 = as.character(adm2@data$NAME_1)
#' adm2@data$NAME_1[adm2@data$NAME_1 == "Trans Nzoia"] = "Trans-Nzoia"
#' adm2@data$NAME_1[adm2@data$NAME_1 == "Elgeyo-Marakwet"] = "Elgeyo Marakwet"
#' 
#' # some Admin-2 areas have the same name
#' adm2@data$NAME_2 = as.character(adm2@data$NAME_2)
#' adm2@data$NAME_2[(adm2@data$NAME_1 == "Bungoma") & 
#'   (adm2@data$NAME_2 == "Lugari")] = "Lugari, Bungoma"
#' adm2@data$NAME_2[(adm2@data$NAME_1 == "Kakamega") & 
#'   (adm2@data$NAME_2 == "Lugari")] = "Lugari, Kakamega"
#' adm2@data$NAME_2[(adm2@data$NAME_1 == "Meru") & 
#'   (adm2@data$NAME_2 == "Igembe South")] = "Igembe South, Meru"
#' adm2@data$NAME_2[(adm2@data$NAME_1 == "Tharaka-Nithi") & 
#'   (adm2@data$NAME_2 == "Igembe South")] = "Igembe South, Tharaka-Nithi"
#' 
#' # The spatial area of unknown 8 is so small, it causes problems unless 
#' # its removed or unioned with another subarea. Union it with neighboring 
#' # Kakeguria:
#' newadm2 = adm2
#' unknown8I = which(newadm2$NAME_2 == "unknown 8")
#' newadm2$NAME_2[newadm2$NAME_2 %in% c("unknown 8", "Kapenguria")] <- "Kapenguria + unknown 8"
#' admin2.IDs <- newadm2$NAME_2
#' 
#' newadm2@data = cbind(newadm2@data, NAME_2OLD = newadm2@data$NAME_2)
#' newadm2@data$NAME_2OLD = newadm2@data$NAME_2
#' newadm2@data$NAME_2 = admin2.IDs
#' newadm2$NAME_2 = admin2.IDs
#' temp <- terra::aggregate(as(newadm2, "SpatVector"), by="NAME_2")
#' 
#' library(sf)
#' temp <- sf::st_as_sf(temp)
#' temp <- sf::as_Spatial(temp)
#' 
#' tempData = newadm2@data[-unknown8I,]
#' tempData = tempData[order(tempData$NAME_2),]
#' newadm2 <- SpatialPolygonsDataFrame(temp, tempData, match.ID = F)
#' adm2 = newadm2
#' 
#' # download 2014 Kenya population density TIF file
#' 
#' githubURL <- paste0("https://github.com/paigejo/SUMMERdata/blob/main/data/", 
#'                     "Kenya2014Pop/worldpop_total_1y_2014_00_00.tif?raw=true")
#' popTIFFilename = paste0(tempDirectory, "/worldpop_total_1y_2014_00_00.tif")
#' if(!file.exists(popTIFFilename)) {
#'   download.file(githubURL,popTIFFilename)
#' }
#' 
#' # load it in
#' pop = terra::rast(popTIFFilename)
#' 
#' east.lim = c(-110.6405, 832.4544)
#' north.lim = c(-555.1739, 608.7130)
#' 
#' require(fields)
#' #' 
#' # Now generate a general population integration table at 5km resolution, 
#' # based on subarea (Admin-2) x urban/rural population totals. This takes 
#' # ~1 minute
#' pop.matKenya <- makePopIntegrationTab(
#'   km.res=5, pop=pop, domain.map.dat=adm0,
#'   east.lim=east.lim, north.lim=north.lim, map.projection=projKenya,
#'   poppa = poppaKenya, poppsub=poppsubKenya, 
#'   area.map.dat = adm1, subarea.map.dat = adm2,
#'   areaNameVar = "NAME_1", subareaNameVar="NAME_2")
#'   
#' out = poppRegionFromPopMat(pop.matKenya, pop.matKenya$area)
#' out
#' poppaKenya
#' 
#' out = poppRegionFromPopMat(pop.matKenya, pop.matKenya$subarea)
#' out
#' poppsubKenya
#' 
#' pop.matKenyaUnstratified = pop.matKenya
#' pop.matKenyaUnstratified$urban = NULL
#' out = poppRegionFromPopMat(pop.matKenyaUnstratified, pop.matKenyaUnstratified$area)
#' out
#' poppaKenya
#' }
#' @return A table of population totals by region
#' 
#' @export
poppRegionFromPopMat = function(pop.mat, regions) {
  stratify.by.urban = ("urban" %in% names(pop.mat)) && !all(is.na(pop.mat$urban))
  
  if(stratify.by.urban) {
    # in this case, generate urban, rural, and overall totals within each region
    out = aggregate(pop.mat$pop, by=list(region=as.character(regions), urban=pop.mat$urban), FUN=sum, drop=FALSE)
    regions = sort(unique(out$region))
    poppr = data.frame(region=regions, popUrb=out[(length(regions) + 1):(2*length(regions)), 3], 
                       popRur=out[1:length(regions), 3])
    poppr$popUrb[is.na(poppr$popUrb)] = 0
    poppr$popRur[is.na(poppr$popRur)] = 0
    poppr$popTotal = poppr$popUrb + poppr$popRur
  } else {
    # in this case, only generate overall totals within each region
    out = aggregate(pop.mat$pop, by=list(region=as.character(regions)), FUN=sum, drop=FALSE)
    regions = sort(unique(out$region))
    poppr = data.frame(region=regions, popTotal=out[1:length(regions), 2])
    poppr$popTotal[is.na(poppr$popTotal)] = 0
  }
  
  poppr
}

#' 
#' `r lifecycle::badge("experimental")`
#' 
#' Set thresholds of population density for urbanicity classifications 
#' within each region of the given type 
#' 
#' @param pop.mat pixellated population density data frame with variables 
#' region.type and `pop`
#' @param poppr A table with population totals by region of the given type 
#' (e.g. poppa or poppsub from \code{\link{makePopIntegrationTab}})
#' @param region.type The variable name from poppr giving the region names. 
#' Defaults to "area"
#' 
#' @details Thresholds are set based on that region's percent population 
#' urban. Intended as a helper function of \code{\link{makePopIntegrationTab}}. 
#' 
#' @author John Paige
#' 
#' @seealso \code{\link{makePopIntegrationTab}}
#' 
#' @examples
#' \dontrun{
#' data(kenyaPopulationData)
#' 
#' #' # download Kenya GADM shapefiles from SUMMERdata github repository
#' githubURL <- "https://github.com/paigejo/SUMMERdata/blob/main/data/kenyaMaps.rda?raw=true"
#' tempDirectory = "~/"
#' mapsFilename = paste0(tempDirectory, "/kenyaMaps.rda")
#' if(!file.exists(mapsFilename)) {
#'   download.file(githubURL,mapsFilename)
#' }
#' 
#' # load it in
#' out = load(mapsFilename)
#' out
#' kenyaMesh <- fmesher::fm_as_fm(kenyaMesh)
#' adm1@data$NAME_1 = as.character(adm1@data$NAME_1)
#' adm1@data$NAME_1[adm1@data$NAME_1 == "Trans Nzoia"] = "Trans-Nzoia"
#' adm1@data$NAME_1[adm1@data$NAME_1 == "Elgeyo-Marakwet"] = "Elgeyo Marakwet"
#' adm2@data$NAME_1 = as.character(adm2@data$NAME_1)
#' adm2@data$NAME_1[adm2@data$NAME_1 == "Trans Nzoia"] = "Trans-Nzoia"
#' adm2@data$NAME_1[adm2@data$NAME_1 == "Elgeyo-Marakwet"] = "Elgeyo Marakwet"
#' 
#' # some Admin-2 areas have the same name
#' adm2@data$NAME_2 = as.character(adm2@data$NAME_2)
#' adm2@data$NAME_2[(adm2@data$NAME_1 == "Bungoma") & 
#'   (adm2@data$NAME_2 == "Lugari")] = "Lugari, Bungoma"
#' adm2@data$NAME_2[(adm2@data$NAME_1 == "Kakamega") & 
#'   (adm2@data$NAME_2 == "Lugari")] = "Lugari, Kakamega"
#' adm2@data$NAME_2[(adm2@data$NAME_1 == "Meru") & 
#'   (adm2@data$NAME_2 == "Igembe South")] = "Igembe South, Meru"
#' adm2@data$NAME_2[(adm2@data$NAME_1 == "Tharaka-Nithi") & 
#'   (adm2@data$NAME_2 == "Igembe South")] = "Igembe South, Tharaka-Nithi"
#' 
#' # The spatial area of unknown 8 is so small, it causes problems unless 
#' # its removed or unioned with another subarea. Union it with neighboring 
#' # Kakeguria:
#' newadm2 = adm2
#' unknown8I = which(newadm2$NAME_2 == "unknown 8")
#' newadm2$NAME_2[newadm2$NAME_2 %in% c("unknown 8", "Kapenguria")] <- "Kapenguria + unknown 8"
#' admin2.IDs <- newadm2$NAME_2
#' 
#' newadm2@data = cbind(newadm2@data, NAME_2OLD = newadm2@data$NAME_2)
#' newadm2@data$NAME_2OLD = newadm2@data$NAME_2
#' newadm2@data$NAME_2 = admin2.IDs
#' newadm2$NAME_2 = admin2.IDs
#' temp <- terra::aggregate(as(newadm2, "SpatVector"), by="NAME_2")
#' 
#' library(sf)
#' temp <- sf::st_as_sf(temp)
#' temp <- sf::as_Spatial(temp)
#' 
#' tempData = newadm2@data[-unknown8I,]
#' tempData = tempData[order(tempData$NAME_2),]
#' newadm2 <- sp::SpatialPolygonsDataFrame(temp, tempData, match.ID = F)
#' adm2 = newadm2
#' 
#' # download 2014 Kenya population density TIF file
#' 
#' githubURL <- paste0("https://github.com/paigejo/SUMMERdata/blob/main/data/", 
#'                     "Kenya2014Pop/worldpop_total_1y_2014_00_00.tif?raw=true")
#' popTIFFilename = paste0(tempDirectory, "/worldpop_total_1y_2014_00_00.tif")
#' if(!file.exists(popTIFFilename)) {
#'   download.file(githubURL,popTIFFilename)
#' }
#' 
#' # load it in
#' pop = terra::rast(popTIFFilename)
#' 
#' east.lim = c(-110.6405, 832.4544)
#' north.lim = c(-555.1739, 608.7130)
#' 
#' require(fields)
#' 
#' data(kenyaPopulationData)
#' 
#' # Now generate a general population integration table at 5km resolution, 
#' # based on subarea (Admin-2) x urban/rural population totals. This takes 
#' # ~1 minute
#' pop.matKenya <- makePopIntegrationTab(
#'   km.res=5, pop=pop, domain.map.dat=adm0,
#'   east.lim=east.lim, north.lim=north.lim, map.projection=projKenya,
#'   poppa = poppaKenya, poppsub=poppsubKenya, 
#'   area.map.dat = adm1, subarea.map.dat = adm2,
#'   areaNameVar = "NAME_1", subareaNameVar="NAME_2")
#' 
#' out = setThresholdsByRegion(pop.matKenya, poppaKenya)
#' out
#' 
#' out = setThresholdsByRegion(pop.matKenya, poppsubKenya, region.type="subarea")
#' out
#' }
#' 
#' @return A list of region names and their urbanicity thresholds in population density
#' 
#' @export
setThresholdsByRegion = function(pop.mat, poppr, region.type="area") {
  
  getRegionThresh = function(regionName) {
    # do the setup
    thisRegion = as.character(pop.mat[[region.type]]) == regionName
    thisPop = pop.mat$pop[thisRegion]
    thisTot = sum(thisPop, na.rm=TRUE)
    pctUrb = poppr$pctUrb[poppr[[region.type]] == regionName]/100
    pctRural = 1 - pctUrb
    
    # must round to avoid numerical inaccuracies
    if(round(pctUrb, 10) == 1) {
      return(-Inf)
    } else if(round(pctUrb, 10) == 0) {
      return(Inf)
    }
    
    # calculate threshold by integrating ecdf via sorted value cumulative sum
    sortedPop = sort(thisPop)
    cumsumPop = cumsum(sortedPop)
    threshI = match(1, cumsumPop >= thisTot*pctRural)
    if((threshI != 1) && (threshI != length(thisPop))) {
      thresh = sortedPop[threshI]
    } else {
      # make sure not all pixels are urban or all are rural
      if(threshI == 1) {
        thresh = mean(c(sortedPop[1], sortedPop[2]), na.rm=TRUE)
      } else {
        thresh = mean(c(sortedPop[length(thisPop)], sortedPop[length(thisPop)-1]), na.rm=TRUE)
      }
    }
    
    thresh
  }
  
  # compute threshold for each area
  regions = poppr[[region.type]]
  threshes = sapply(regions, getRegionThresh)
  
  list(regions=regions, threshes=threshes)
}



