#' Log-Likelihood Function for 3D Copula Model
#'
#' Computes the negative log-likelihood of a 3-dimensional copula model with a
#' time-varying copula structure.
#'
#' @param params Numeric vector, model parameters.
#' @param u1 Numeric vector (length `n_train`), pseudo-observations for margin 1.
#' @param u2 Numeric vector (length `n_train`), pseudo-observations for margin 2.
#' @param u3 Numeric vector (length `n_train`), pseudo-observations for margin 3.
#' @param X_t Numeric matrix (`n_train x M`), risk factors affecting copula parameters.
#' @param z1 Numeric matrix (`n_train x M`), observed data for margin 1.
#' @param z2 Numeric matrix (`n_train x M`), observed data for margin 2.
#' @param z3 Numeric matrix (`n_train x M`), observed data for margin 3.
#' @param copula Character, specifying the copula type: "Clayton", "Frank", "Gumbel", "Joe", or "Gaussian".
#' @return The negative log-likelihood value for optimization.
#' @importFrom stats cor rnorm dnorm
#' @importFrom copula claytonCopula frankCopula gumbelCopula joeCopula normalCopula dCopula
#' @export
#'
#' @examples test_ll_3d <- log_likelihood_noGEV_3d(init_params_full,
#'                                                 u[[1]],
#'                                                 u[[2]],
#'                                                 u[[3]],
#'                                                 (z_train[[1]] + z_train[[2]] + z_train[[3]])/3,
#'                                                 z_train[[1]],
#'                                                 z_train[[2]],
#'                                                 z_train[[3]],
#'                                                 "Gaussian")
#'
log_likelihood_noGEV_3d <- function(params, u1, u2, u3, X_t, z1, z2, z3, copula) {

  result <- try({# Parameters
    omega <- params[1]
    alpha <- params[2]
    gamma <- params[3:(3+ncol(z1)-1)]

    n <- length(u1)
    if(copula == "Gaussian"){
      theta <- matrix(NA, nrow = n, ncol = 3)
    } else{
      theta <- numeric(n)
    }

    X_12 <- mapply(pmax, as.data.frame(z1), as.data.frame(z2))
    X_13 <- mapply(pmax, as.data.frame(z1), as.data.frame(z3))
    X_23 <- mapply(pmax, as.data.frame(z2), as.data.frame(z3))


    # Initial theta based on copula type
    if (copula == "Clayton") {
      #theta[1] <- abs(omega + alpha * clayton.theta(cor(u1, u2, method = "kendall")) +  sum(as.numeric(gamma) * colMeans(X_t)))
      theta[1] <- abs(omega + alpha * clayton.theta(mean(cor(cbind(u1, u2, u3), method = "kendall"))) +  sum(as.numeric(gamma) * colMeans(X_t)))
    } else if (copula == "Frank") {
      #theta[1] <- abs(omega + alpha * frank.theta(mean(cor(cbind(u1, u2, u3, u4), method = "kendall"))) + gamma * mean(X_t))
      theta[1] <- abs(omega + alpha * 3.58 +  sum(as.numeric(gamma) * colMeans(X_t)))
    } else if (copula == "Gumbel") {
      theta[1] <- abs(omega + alpha * GH.theta(mean(cor(cbind(u1, u2, u3), method = "kendall"))) +  sum(as.numeric(gamma) * colMeans(X_t))) + 1
    } else if (copula == "Joe") {
      #theta[1] <- abs(omega + alpha * joe.theta(mean(cor(cbind(u1, u2, u3, u4), method = "kendall"))) + gamma * mean(X_t)) + 1
      theta[1] <- abs(omega + alpha * 2.003 +  sum(as.numeric(gamma) * colMeans(X_t)))+ 1
    } else if (copula == "Gaussian") {
      theta[1,1] <- tanh(omega + alpha * cor(u1, u2, method = "pearson") +  sum(as.numeric(gamma) * X_12[1,]))
      theta[1,2] <- tanh(omega + alpha * cor(u1, u3, method = "pearson") +  sum(as.numeric(gamma) * X_13[1,]))
      theta[1,3] <- tanh(omega + alpha * cor(u2, u3, method = "pearson") +  sum(as.numeric(gamma) * X_23[1,]))
      #theta[1] <- tanh(omega + alpha * mean(cor(cbind(u1, u2, u3, u4), method = "pearson")) +  sum(as.numeric(gamma) * colMeans(X_t)))
    } else {
      stop("Copula provided is not valid")
    }

    # Initialize log-likelihood
    ll <- 0

    # Loop over time
    for (t in 2:n) {

      # Update conditional copula parameter theta
      if (copula == "Clayton") {
        theta[t] <- dynamic.theta.clayton(params = c(omega, alpha, gamma), theta[t - 1], X_t[t,])
        cop <- claytonCopula(param = theta[t], dim = 3)
      } else if (copula == "Frank") {
        theta[t] <- dynamic.theta.frank(params = c(omega, alpha, gamma), theta[t - 1], X_t[t,])
        cop <- frankCopula(param = theta[t], dim = 3)
      } else if (copula == "Gumbel") {
        theta[t] <- dynamic.theta.gumbel(params = c(omega, alpha, gamma), theta[t - 1], X_t[t,])
        cop <- gumbelCopula(param = theta[t], dim = 3)
      } else if (copula == "Joe") {
        theta[t] <- dynamic.theta.joe(params = c(omega, alpha, gamma), theta[t - 1], X_t[t,])
        cop <- joeCopula(param = theta[t], dim = 3)
      } else if (copula == "Gaussian") {
        theta[t,1] <- dynamic.rho(params = c(omega, alpha, gamma), theta[t - 1,1], X_12[t,])
        theta[t,2] <- dynamic.rho(params = c(omega, alpha, gamma), theta[t - 1,2], X_13[t,])
        theta[t,3] <- dynamic.rho(params = c(omega, alpha, gamma), theta[t - 1,3], X_23[t,])
        cor_matrix_offdiag <- c(theta[t,1],theta[t,2],theta[t,3])
        #theta[t] <- dynamic.rho(params = c(omega, alpha, gamma), theta[t - 1], X_t[t,])
        cop <- normalCopula(param = cor_matrix_offdiag, dim = 3, dispstr = "un")
      }

      # Copula contribution
      copula_density <- log(dCopula(cbind(u1[t], u2[t], u3[t]), copula = cop))

      # Update log-likelihood
      ll <- ll + copula_density
    }

    # Print params and log-likelihood for debugging
    #print(c(params, ll))

    # Return negative log-likelihood for optimization
    return(-ll)
  },silent = TRUE)

  # Handle errors
  if (inherits(result, "try-error")) {
    message("Error encountered in log-likelihood calculation: ", result)
    return(Inf)  # Return a penalty value to continue optimization
  }

  return(result)
}
