#' Identification of the Optimal Sampling Effort
#'
#' Estimates the sampling effort at which the improvement in precision (MultSE) per additional sampling unit
#' becomes sub-optimal or redundant, based on predefined cut-off thresholds.
#'
#' @param xx A data frame generated by \code{\link{summary_ssp}}.
#' @param multi.site Logical. Indicates whether multiple sites were simulated.
#' @param c1 First cut threshold. Default is 10\% improvement over the highest MultSE.
#' @param c2 Second cut threshold. Default is 5\% improvement over the highest MultSE.
#' @param c3 Third cut threshold. Default is 2.5\% improvement over the highest MultSE.
#'
#' @details
#' Sampling efforts between the minimum (e.g. 2 samples) and \code{c1} represent the necessary effort to achieve acceptable precision.
#' Efforts between \code{c1} and \code{c2} reflect sub-optimal gains, and those between \code{c2} and \code{c3} are considered optimal.
#' Beyond \code{c3}, any additional effort results in marginal improvements in MultSE and may be considered redundant.
#' This classification helps support cost-benefit decisions in ecological survey design (see Underwood, 1990).
#' If \code{c3} is not reached within the simulated range, the maximum available effort is returned with a warning.
#'
#' @return A vector or matrix indicating the sampling sizes corresponding to each cut-off point.
#'
#' @note
#' The cut-off thresholds are arbitrary and should be adjusted based on the ecological question and resource availability.
#' In some cases, \code{c3} may not be reached within the range of simulated sampling efforts.
#'
#' @references
#' Underwood, A. J. (1990). Experiments in ecology and management: Their logics, functions and interpretations. Australian Journal of Ecology, 15, 365–389.
#'
#' Guerra-Castro, E.J., Cajas, J.C., Simões, N., Cruz-Motta, J.J., & Mascaró, M. (2021). SSP: an R package to estimate sampling effort in studies of ecological communities. Ecography 44(4), 561-573. doi: \doi{10.1111/ecog.05284}
#'
#' @seealso \code{\link{plot_ssp}}, \code{\link{summary_ssp}}, \code{\link{sampsd}}
#'
#' @examples
#' ## Single site example
#' data(micromollusk)
#' par.mic <- assempar(data = micromollusk, type = "P/A", Sest.method = "average")
#' sim.mic <- simdata(par.mic, cases = 2, N = 20, sites = 1)
#' sam.mic <- sampsd(dat.sim = sim.mic,
#'                   Par = par.mic,
#'                   transformation = "P/A",
#'                   method = "jaccard",
#'                   n = 10,
#'                   m = 1,
#'                   k = 3)
#' summ.mic <- summary_ssp(results = sam.mic, multi.site = FALSE)
#' opt.mic <- ioptimum(xx = summ.mic, multi.site = FALSE)
#'
#' ## See the full multi-site workflow in the vignette
#'
#' @export

ioptimum<-function(xx, multi.site = TRUE, c1 = 10, c2 =5, c3= 2.5){

  ##MultSE for multiple sites
  if (multi.site == TRUE) {
    #subsetting data in sites and samples
    mse.sites<- xx[xx$sv=="sites", c(1,7)]
    mse.samples<- xx[xx$sv=="samples", c(1,7)]

    #Find the minimun sampling effort for improvement of c1, c2 and c3 for each sampling unit.
    #matrix to store values
    sample.cut<-matrix(nrow=2, ncol = 3)
    rownames(sample.cut)<-c("sites", "Samples")
    colnames(sample.cut)<-c("c1", "c2", "c3")

    #cut points for sites
    s.c1<-mse.sites[mse.sites$der<=c1, 1]
    s.c2<-mse.sites[mse.sites$der<=c2, 1]
    s.c3<-mse.sites[mse.sites$der<=c3, 1]

    if (is.na(s.c1[2])){
      warning("Sites: the required optimizations were not achieved, instead, the maximum number of sites sampled with 'sampsd' is presented. Increase the 'm' value in 'sampsd' to achieve the specified optimization or relax the optimization values in arguments c1, c2 and c3", call. = FALSE)
      s.c3 <- s.c2 <- s.c1 <- max(mse.sites[,1])
    }

    if (is.na(s.c2[2])){
      warning("Sites: the required optimizations (c2, c3) were not achieved, instead, the maximum number of sites sampled with 'sampsd' is presented. Increase the 'm' value in 'sampsd' to achieve the specified optimization values or relax the optimization in arguments c2 and c3", call. = FALSE)
      s.c3 <- s.c2 <- max(mse.sites[,1])
    }

    if (is.na(s.c3[2])){
      warning("Sites: the required optimizations (c3) was not achieved, instead, the maximum number of sites sampled with 'sampsd' is presented. Increase the 'm' value in 'sampsd' to achieve the specified optimization or relax the optimization in argument c3", call. = FALSE)
      s.c3<-max(mse.sites[,1])
    }


    #cut points for samples
    n.c1<-mse.samples[mse.samples$der<=c1, 1]
    n.c2<-mse.samples[mse.samples$der<=c2, 1]
    n.c3<-mse.samples[mse.samples$der<=c3, 1]

    if (is.na(n.c1[2])){
      warning("Samples: the required optimizations were not achieved, instead, the maximum effort generated with 'sampsd' is presented. Increase the 'n' value in 'sampsd' to achieve the specified optimization or relax the optimization values in arguments c1, c2 and c3", call. = FALSE)
      n.c3 <- n.c2 <- n.c1 <- max(mse.samples[,1])
      }

    if (is.na(n.c2[2])){
      warning("Samples: the required optimizations (c2, c3) were not achieved, instead, the maximum effort generated with 'sampsd' is presented. Increase the 'n' value in 'sampsd' to achieve the specified optimization values or relax the optimization in arguments c2 and c3", call. = FALSE)
      n.c3 <- n.c2 <- max(mse.samples[,1])
    }

    if (is.na(n.c3[2])){
      warning("Samples: the required optimization was not achieved, instead, the maximum effort generated with 'sampsd' is presented. Increase the 'n' value in 'sampsd' to achieve the specified optimization or relax the optimization in argument c3", call. = FALSE)
      n.c3<-max(mse.samples[,1])
      }

    #add values to the matrix
    sample.cut[1,1]<- s.c1[2]
    sample.cut[1,2]<- s.c2[2]
    sample.cut[1,3]<- s.c3[2]
    sample.cut[2,1]<- n.c1[2]
    sample.cut[2,2]<- n.c2[2]
    sample.cut[2,3]<- n.c3[2]
    return(sample.cut)

  }
  ##MultSE for a single site
  if (multi.site == FALSE){
    n.c1<-xx[xx$der<=c1,1]
    n.c2<-xx[xx$der<=c2,1]
    n.c3<-xx[xx$der<=c3,1]

    if (is.na(n.c1[2])){
      warning("The required optimizations were not achieved, instead, the maximum effort generated with 'sampsd' is presented. Increase the 'n' value in 'sampsd' to achieve the specified optimization or relax the optimization values in arguments c1, c2 and c3", call. = FALSE)
      n.c3 <- n.c2 <- n.c1 <- max(xx[,1])
      sample.cut<-c(n.c1[1], n.c2[1], n.c3[1])
      names(sample.cut)<-c("c1", "c2", "c3")
      return(sample.cut)

    }

    if (is.na(n.c2[2])){
      warning("The required optimizations (c2, c3) were not achieved, instead, the maximum effort generated with 'sampsd' is presented. Increase the 'n' value in 'sampsd' to achieve the specified optimization values or relax the optimization in arguments c2 and c3", call. = FALSE)
      n.c3 <- n.c2 <- max(xx[,1])
      sample.cut<-c(n.c1[2], n.c2, n.c3)
      names(sample.cut)<-c("c1", "c2", "c3")
      return(sample.cut)
      }

    if (is.na(n.c3[2])){
      warning("The required optimization (c3) was not achieved, instead, the maximum effort generated with sampsd is presented. Increase the 'n' value in 'sampsd' to achieve the specified optimization or relax the optimization in argument c3", call. = FALSE)
      n.c3<-max(xx[,1])
      sample.cut<-c(n.c1[2], n.c2[2], n.c3)
      names(sample.cut)<-c("c1", "c2", "c3")
      return(sample.cut)

    } else {

    sample.cut<-c(n.c1[2], n.c2[2], n.c3[2])
    names(sample.cut)<-c("c1", "c2", "c3")
    return(sample.cut)
    }
  }
}
