#' Make Predictions from a `quanda` Object
#'
#' Produces fitted values for new predictor data using a fitted `quanda()` object.
#' @importFrom methods rbind2 cbind2
#' @importFrom stats predict
#' @param object Fitted `quanda()` object from which predictions are to be derived.
#' @param newx Matrix of new predictor values for which predictions are desired.
#'   This must be a matrix and is a required argument.
#' @param type Type of prediction required. Type `"class"` produces the predicted binary class labels and
#' type `"loss"` returns the fitted values. Default is \code{"class"}.
#' @param ... Not used.
#' @seealso \code{\link{quanda}}
#'
#' @return Numeric vector of length \code{n_new}.
#'
#' @method predict quanda
#' @examples
#' data(breast)
#' X <- as.matrix(X)
#' y <- as.numeric(as.character(y))
#' y[y==-1]=0
#' \donttest{fit <- quanda(X, y)}
#' @export

predict.quanda <- function(object, newx, type=c("class", "loss"), ...) {
  type <- match.arg(type)
  if (missing(newx) || is.null(newx)) stop("newx is required.")
  nbeta <- object$beta
  nfit <- as.matrix(as.matrix(cbind2(1, newx)) %*% nbeta)-1
  switch(type, loss=nfit, class=ifelse(nfit > 0, 1, 0))
}