% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nonp_limits.r
\name{nonp_limits}
\alias{nonp_limits}
\title{Nonparametric Limits}
\usage{
nonp_limits(X, RR=TRUE, perc=0.95, cover=0.9)
}
\arguments{
\item{X}{the data set whose reference limit is sought.}

\item{RR}{\emph{optional} (default of TRUE) - the way percentiles are defined:
\itemize{
\item FALSE -- use the Weibull limit i/(n+1),
\item TRUE  -- use the Hazen limit (i-0.5)/n.
}}

\item{perc}{\emph{optional} (default of 0.95) - the two-sided probability.}

\item{cover}{\emph{optional} (default of 0.9) - the confidence level of the CI for the reference limit.}
}
\value{
A list containing the following components:

\item{lower}{the lower limit and confidence interval}
\item{upper}{the upper limit and confidence interval}
\item{a}{the index of the order statistic defining the lower limit of the CI}
\item{b}{the index of the order statistic defining the upper limit of the CI}
\item{coverage}{the actual confidence level the interval achieves}
}
\description{
Calculate shortest nonparametric reference limits, at given confidence level.
}
\details{
The reference limits are estimated as lower and upper percentiles of the sample.
There are many ways of defining sample percentiles of which the code has
two options – the Hazen limit (preferred) and the Weibull limit
(which has historical precedents).

The confidence limits on these percentiles come from standard binomial methodology.
\itemize{
\item Each interval is a pair of order statistics from the sample.
\item The codes find the order statistics with the required 90\% coverage
but use order statistics with indices as close together as possible.
\item An implication of this is that the 90\% confidence interval on each
reference limit has a total tail area less than 10\% but does not
necessarily have less than 5\% in each tail.
}

If the sample size is too small (<91 for the default settings), confidence
limits can not be computed and will be returned as NA.
If it is even smaller (<21 for the default settings) the estimated
reference limits will also be returned as NA.
}
\examples{
# parameters
mu    <- 40
sigma <- 10
n     <- 120

# replicable randomization
set.seed(1069)
X     <- mu + sigma*rnorm(n)

# evaluate and review
perc_used = 0.95
nonp_results <- nonp_limits(X, perc=perc_used)
cat("\nThe reference limits are the", 100*(1-perc_used)/2, "th percentile",
    signif(nonp_results$lower[1],5), "and the", 100*(1+perc_used)/2, "th percentile",
    signif(nonp_results$upper[1],5), ".\n")

cat("\nAnd with ", round(100*nonp_results$coverage,2),"\% confidence, the lower limit is between",
    signif(nonp_results$lower[2],5), "and", signif(nonp_results$lower[3],5),
    ";\n while the upper limit is between",
    signif(nonp_results$upper[2],5),"and",signif(nonp_results$upper[3],5),".\n\n")

}
\references{
Horn PS, Peske AJ (2005). Reference intervals: a user’s guide. Washington (DC): AACC Press.
}
\author{
Douglas M. Hawkins, Jessica J. Kraker \href{mailto:krakerjj@uwec.edu}{krakerjj@uwec.edu}
}
