#' Heatmap
#'
#' Generate a heatmap to show different activity intensities (in MET) at different time of different days.
#' @param data  A csv file for one participant with multiple days' activity records from SenseWear. Data format refers to provided \code{sampledata}.
#' @param a The desired cutpoints of METs. Lower and upper limits must be specified. E.g. \code{a=c(0,3,5,7)}. 0 and 7 are the lower and upper limit, respectively.
#' @param category Default is FALSE which means treating METs as continuous. \code{category=TRUE} and a valid cutpoints \code{a} will categorize METs by \code{a}. If \code{category=TRUE} while no \code{a} is specified, METs will be treated as continuous.
#' @return \code{graph}	A heatmap generated by \code{ggplot} with x axis Time and y axis Date
#' @importFrom  ggplot2 ggplot aes geom_tile scale_x_discrete scale_fill_manual theme_bw
#' @examples
#' #Continuous METs
#' Heatmap(sampledata);
#' #Categorical METs with cutpoint 0,3,5,7
#'  Heatmap(sampledata,c(0,3,5,7),category=TRUE)

#' @export


Heatmap=function(data,a,category=FALSE){

data1=data[,c(1,23)]
colnames(data1)=c("Time1","METs")
METs=data1$METs
Time=0
Date=0
data1=cbind(data1,Time,Date)
data1$Time=as.POSIXct(data1$Time1)
data1$Date <- as.Date(data1$Time1, format = "%Y-%m-%d")
lab <- with(data1, paste(format(data1$Time, "%H"), "00", sep = ":"))
data1$Time <- format(data1$Time, "%H:%M")

if (category)
{METs_cat=0
data1=cbind(data1,METs_cat)
data1$METs_cat=cut(data1$METs,breaks=a)

color_palette <- colorRampPalette(c("#31a354","#2c7fb8", "#fcbfb8", "#f03b20"))(length((levels(factor(data1$METs_cat)))))
ggplot(data = data1) +
  geom_tile( aes(x = Time, y = Date, fill = METs_cat))+scale_x_discrete(breaks = c("00:00","06:00","12:00","18:00","23:00"))+ scale_fill_manual(values =color_palette)+theme_bw()
}else{
 ggplot(data = data1) +
  geom_tile( aes(x = Time, y = Date, fill = METs))+scale_x_discrete(breaks = c("00:00","06:00","12:00","18:00","23:00"))+scale_fill_distiller(palette = "Spectral",limits=c(0,6))+  theme_bw()
}
}
