#' Default values passed to RotMat*
#'
#' Given the parameter list and the categorical map this function populates the values of the parameter list accoding to our 'best'
#' known general use case parameters.
#'
#' @param paramList A list (possibly empty), to be populated with a set of default values to be passed to a \code{RotMat*} function.
#' @param split The criterion used for splitting the variable. 'gini': gini impurity index (classification, default),
#'        'entropy': information gain (classification) or 'mse': mean square error (regression).
#' @param dimX An integer denoting the number of columns in the design matrix X.
#' @param weights A vector of length same as \code{data} that are positive weights.(default NULL)
#' @param catLabel A category labels of class \code{list} in predictors. (default NULL, for details see Examples of \code{\link{ODT}})
#'
#' @return Default parameters of the RotMat* function.
#' \itemize{
#' \item \code{dimX} An integer denoting the number of columns in the design matrix X.
#' \item \code{dimProj} Number of variables to be projected, default \code{dimProj="Rand"}: random from 1 to ncol(X).
#' \item \code{numProj} the number of projection directions.(default \code{ceiling(sqrt(dimX))})
#' \item \code{catLabel} A category labels of class \code{list} in prediction variables, for details see Examples of \code{\link{ODRF}}.
#' \item \code{weights} A vector of length same as \code{data} that are positive weights.(default NULL)
#' \item \code{lambda} Parameter of the Poisson distribution (default 1).
#' \item \code{sparsity} A real number in \eqn{(0,1)} that specifies the distribution of non-zero elements in the random matrix.
#' When \code{sparsity}="pois" means that non-zero elements are generated by the p(\code{lambda}) Poisson distribution.
#' \item \code{prob} A probability \eqn{\in (0,1)} used for sampling from.
#' \item \code{randDist} Parameter of the Poisson distribution (default 1).
#' \item \code{split} The criterion used for splitting the variable. 'gini': gini impurity index (classification, default),
#' 'entropy': information gain (classification) or 'mse': mean square error (regression).
#' \item \code{model} Model for projection pursuit. (see \code{\link{PPO}})
#' }
#'
#' @seealso \code{\link{RotMatPPO}} \code{\link{RotMatRand}} \code{\link{RotMatRF}} \code{\link{RotMatMake}}
#'
#' @keywords tree rotation
#' @examples
#' set.seed(1)
#' paramList <- list(dimX = 8, numProj = 3, sparsity = 0.25, prob = 0.5)
#' (paramList <- defaults(paramList, split = "entropy"))
#' @export
defaults <- function(paramList, split = "entropy", dimX = NULL, weights = NULL, catLabel = NULL) {
  # public parameter.
  if (is.null(paramList[["dimX"]])) {
    paramList$dimX <- dimX
  }
  if (is.null(paramList[["dimProj"]])) {
    paramList$dimProj <- "Rand"
  }
  if (is.null(paramList[["catLabel"]])) {
    paramList$catLabel <- catLabel
  }
  if (is.null(paramList[["numProj"]])) {
    #  q<- min(ceiling(length(y)^0.4),ceiling(paramList$p*2/3))
    #  paramList$d <-min(max(5, ceiling(paramList$p/q)),paramList$p)
    # if(NodeRotateFun!="RotMatPPO")
    p <- paramList$dimX - (!is.null(paramList$catLabel)) * (length(unlist(paramList$catLabel)) - length(paramList$catLabel))
    paramList$numProj <- ceiling(sqrt(p))
  }
  if (is.null(paramList[["weights"]])) {
    paramList$weights <- weights
  }
  if (is.null(paramList[["lambda"]])) {
    paramList$lambda <- 1
  }

  # RandRotMat parameter.
  #############################################
  if (is.null(paramList[["sparsity"]])) {
    paramList$sparsity <- ifelse(paramList$dimX >= 10, 3 / paramList$dimX, 1 / paramList$dimX)
  }
  if (is.null(paramList[["prob"]])) {
    paramList$prob <- 0.5
  }
  if (is.null(paramList[["randDist"]])) {
    paramList$randDist <- c("Binary", "Norm", "Uniform")[1]
  }

  # RotMatPPO parameter.
  ###########################################
  if (is.null(paramList[["split"]])) {
    paramList$split <- split # method='g-classification'
  }
  if (is.null(paramList[["model"]])) {
    paramList$model <- "PPR"
  }

  return(paramList)
}
