#' Order Neighborhood for MOEA/D
#'
#' Calculates the ordering of competing solutions for each subproblem in the
#' MOEA/D, based on their scalarized performance and violation values.
#'
#' This routine receives a matrix of scalarized performance values (returned by
#' [scalarize_values()]), a neighborhood matrix, and the list of violation
#' values for the candidate and incumbent populations. It calculates the
#' preference order of the candidates for each neighborhood based on the
#' performance values and constraint handling method.
#'
#' The list of available constraint handling methods can be generated using
#' [get_constraint_methods()].
#'
#' @param bigZ Matrix of scalarized performance values by neighborhood,
#' generated by [scalarize_values()]
#' @param B Neighborhood matrix, generated by [define_neighborhood()].
#' @param V List object containing information about the constraint violations
#' of the _candidate solutions_, generated by [evaluate_population()]
#' @param Vt List object containing information about the constraint violations
#' of the _incumbent solutions_, generated by [evaluate_population()]
#' @param constraint list containing the parameters defining the constraint
#' handling method. See Section `Constraint Handling` of the [moead()]
#' documentation for details.
#'
#' @return `[N x (T+1)]` matrix of preference indexes. Each row contains
#' the T indexes of the candidate solutions in the neighborhood of
#' a given subproblem, plus a value (column T+1) for the incumbent solution of
#' that subproblem, in an order defined by the constraint handling method
#' specified in \code{moead.env$constraint}.
#'
#' @export
#'
#' @section References:
#' F. Campelo, L.S. Batista, C. Aranha (2020): The {MOEADr} Package: A
#' Component-Based Framework for Multiobjective Evolutionary Algorithms Based on
#' Decomposition. Journal of Statistical Software \doi{10.18637/jss.v092.i06}\cr
#'

order_neighborhood <- function(bigZ, B, V, Vt, constraint)
{

  # ========== Assert that moead.env has all necessary variables
  assertthat::assert_that(
    nrow(bigZ) == ncol(B) + 1,
    ncol(bigZ) == nrow(B))
  # ==========

  # If we don't have constraints, just sort indexes by bigZ
  if (is.null(V))
  {
    # Get the selection matrix for all neighborhoods using only bigZ
    sel.indx <- t(apply(bigZ,
                        MARGIN = 2,
                        FUN    = function (X) {
                          unlist(as.matrix(sort.int(X,
                                                    index.return = TRUE))[2])}))
    # Code snipped for getting vector of sorting indexes from
    # https://joelgranados.com/2011/03/01/r-finding-the-ordering-index-vector/
  }
  else
  {
    # calculate the penalty matrix of the neighborhoods and incumbent solution,
    # using the same process used to calculate bigZ (see scalarize_values)
    bigV <- t(cbind(matrix(V$v[B],
                           dim(B)),
                    Vt$v))

    # use constraint handler function to calculate selection matrix
    opname       <- paste0("constraint_", constraint$name)

    # Update list of function inputs
    ord.args      <- constraint
    ord.args$name <- NULL
    ord.args$B    <- B
    ord.args$bigZ <- bigZ
    ord.args$bigV <- bigV
    ord.args$V    <- V$v
    ord.args$Vt   <- Vt$v

    # call constraint handling method
    sel.indx <- do.call(opname,
                        args = ord.args)
  }

  return(sel.indx)
}
