#' Printing outputs of a LongDecompHE object
#' @name print.LongDecompHE
#' @aliases print.LongDecompHE
#' @param x a LongDecompHE object
#' @param ... further arguments
#' @importFrom stats printCoefmat
#' @return No return value, called for prints model details to the console.
#' @export
print.LongDecompHE <- function(x,...) {
  if (identical(x$copula, "Attribution")) return(invisible(x))

  else if (!is.null(x$m.dist)){
    cat("Copula:  ",x$copula,"\n")
    cat("Margin:  ",x$m.dist,"\n\n")
    if (dim(x$summary)[2] > 1){
      printCoefmat(x$summary, P.values = TRUE, has.Pvalue = TRUE)
      cat("(The Wald tests are testing whether each coefficient is 0)\n")
    } else {
      printCoefmat(x$summary, P.values = FALSE, has.Pvalue = FALSE)
    }
    cat("\nFinal llk:  ", x$llk, "\n")
    if (!is.null(x$code) && x$code == 0) {
      cat("Convergence is completed successfully\n")
    }
  } else {
    cat("Copula:  ",x$copula,"\n")
    cat("Margin:  semiparametric\n\n")
    if (dim(x$summary)[2] > 1){
      printCoefmat(x$summary, P.values = TRUE, has.Pvalue = TRUE)
      cat("(The Wald tests are testing whether each coefficient is 0)\n")
    } else {
      printCoefmat(x$summary, P.values = FALSE, has.Pvalue = FALSE)
    }
    cat("\nFinal llk:  ", x$llk, "\n")
    if (!is.null(x$code) && x$code == 0) {
      cat("Convergence is completed successfully\n")
    }
  }
}

#' Summarizing outputs of a LongDecompHE object
#'
#' #' This function prints or returns a summary of results from an attribution or decomposition
#' analysis using the \code{LongDecompHE} framework.
#'
#' @name summary.LongDecompHE
#' @aliases summary.LongDecompHE
#' @param object a LongDecompHE object
#' @param ... further arguments
#'
#' @return
#' If the input is an attribution or decomposition object (e.g., from \code{Attribution_sullivan} or \code{Decomp_sullivan}),
#' the function prints summary tables of contributions to disability and death and returns the object invisibly.
#' Otherwise, it returns a list with model diagnostics including:
#' \itemize{
#'   \item \code{copula}: the copula family used.
#'   \item \code{m.dist}: marginal distribution type.
#'   \item \code{summary}: coefficient estimates and standard errors.
#'   \item \code{llk}: log-likelihood.
#'   \item \code{AIC}: Akaike Information Criterion.
#'   \item \code{code}: convergence code.
#' }
#' The returned object has class \code{"summary.LongDecompHE"}.
#'
#' @export
summary.LongDecompHE <- function(object, ...) {
  if (identical(object$copula, "Attribution_sullivan")) {
    cat("Summary of Attribution Results:\n\n")

    if (!is.null(object$Relative_Contributions_1)) {
      cat(">> Relative Contributions for Disability:\n")
      print(round(object$Relative_Contributions_1, 4))
      cat("\n")
    }

    if (!is.null(object$Relative_Contributions_2)) {
      cat(">> Relative Contributions for Death:\n")
      print(round(object$Relative_Contributions_2, 4))
      cat("\n")
    }

    if (!is.null(object$Absolute_Contributions_1)) {
      cat(">> Absolute Contributions for Disability:\n")
      print(round(object$Absolute_Contributions_1, 4))
      cat("\n")
    }

    if (!is.null(object$Absolute_Contributions_2)) {
      cat(">> Absolute Contributions for Death:\n")
      print(round(object$Absolute_Contributions_2, 4))
      cat("\n")
    }

    invisible(object)
  }
  else if (identical(object$copula, "Decomposition_sullivan")) {
    cat("Summary of Decomposition Results:\n\n")

    if (!is.null(object$total_effect)) {
      cat(">> Total Effect for Cohort2 - Cohort1:\n")
      print(round(object$total_effect, 4))
      cat("\n")
    }

    if (!is.null(object$mortality_effect)) {
      cat(">> Mortality Effect for Cohort2 - Cohort1:\n")
      print(round(object$mortality_effect, 4))
      cat("\n")
    }

    if (!is.null(object$disability_effect)) {
      cat(">> Disability Effect for Cohort2 - Cohort1:\n")
      print(round(object$disability_effect, 4))
      cat("\n")
    }

    invisible(object)
  }
  else {
    res <- list(copula = object$copula, m.dist = object$m.dist, summary = object$summary,
                llk = object$llk, AIC = object$AIC, code = object$code)
    class(res) <- "summary.LongDecompHE"
    res
  }
}
