% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_mature.R
\name{calc_mature}
\alias{calc_mature}
\title{Calculate the proportion of mature individuals}
\usage{
calc_mature(Lmat, nfish, mid, kappa, sc_Linf, eps = 1e-05, force_mature = TRUE)
}
\arguments{
\item{Lmat}{A numeric vector of length \code{nfish} representing the length at which 50\% of individuals are mature for each species.}

\item{nfish}{A numeric value representing the number of species in the model.}

\item{mid}{A numeric vector of length \code{nfish} representing the mid-point of the length classes in the model.}

\item{kappa}{A numeric vector of length \code{nfish} representing the rate of change from immaturity to maturity for each species.}

\item{sc_Linf}{A numeric vector of length \code{nsc} representing the length class at which each species reaches its asymptotic length.}

\item{eps}{A numeric value specifying a numerical offset. The default is \code{1e-5}.}

\item{force_mature}{A logical statement indicating whether to force all fish in the largest length class to be mature. The default is \code{TRUE}.}
}
\value{
A matrix with dimensions \code{nsc} and \code{nfish} and elements in the range 0-1 representing the proportion of individuals that are mature for each species and length class.
}
\description{
Calculates the proportion of individuals that are mature for each species and length class.
}
\details{
The proportion of individuals in the \code{j}th length class of the \code{i}th species that are mature is described by a logistic model

\code{1/(1+exp(-kappa[i]*(mid[j]-Lmat[i])))}
}
\examples{
# Set up the inputs to the function - species-independent parameters
nfish <- nrow(NS_par)
nsc <- 32
maxsize <- max(NS_par$Linf)*1.01 # the biggest size is 1\% bigger than the largest Linf
l_bound <- seq(0, maxsize, maxsize/nsc); l_bound <- l_bound[-length(l_bound)]
u_bound <- seq(maxsize/nsc, maxsize, maxsize/nsc)
mid <- l_bound+(u_bound-l_bound)/2

# Set up the inputs to the function - species-specific parameters
Linf <- NS_par$Linf # the von-Bertalanffy asymptotic length of each species (cm).
W_a <- NS_par$W_a # length-weight conversion parameter.
W_b <- NS_par$W_b # length-weight conversion parameter.
k <- NS_par$k # the von-Bertalnaffy growth parameter.
Lmat <- NS_par$Lmat # the length at which 50\\\% of individuals are mature (cm).

# Get phi_min
tmp <- calc_phi(k, Linf, nsc, nfish, u_bound, l_bound, calc_phi_min=FALSE,
                  phi_min=0.1) # fixed phi_min
phi <- tmp$phi
phi_min <- tmp$phi_min

# Calculate growth increments
tmp <- calc_ration_growthfac(k, Linf, nsc, nfish, l_bound, u_bound, mid, W_a, W_b, phi_min)
sc_Linf <- tmp$sc_Linf

# Calculate the proportion of mature individuals
mature <- calc_mature(Lmat, nfish, mid, kappa=rep(10, nfish), sc_Linf)
}
\references{
Hall, S. J., Collie, J. S., Duplisea, D. E., Jennings, S., Bravington, M., & Link, J. (2006). A length-based multispecies model for evaluating community responses to fishing. \emph{Canadian Journal of Fisheries and Aquatic Sciences}, 63(6):1344-1359.
}
