# JapanAPIs - Access Japanese Data via Public APIs and Curated Datasets
# Version 0.1.1
# Copyright (c) 2025 Renzo Caceres Rossi
# Licensed under the MIT License.
# See the LICENSE file in the root directory for full license text.

# get_japan_child_mortality

library(testthat)

test_that("get_japan_child_mortality() returns a tibble with expected structure", {
  skip_on_cran()

  result <- get_japan_child_mortality()

  # Structure
  expect_s3_class(result, "tbl_df")
  expect_equal(ncol(result), 4)

  # Column names
  expect_named(result, c("indicator", "country", "year", "value"))

  # Column types
  expect_type(result$indicator, "character")
  expect_type(result$country, "character")
  expect_type(result$year, "integer")
  expect_type(result$value, "double")
})

test_that("get_japan_child_mortality() returns correct dimensions and years", {
  skip_on_cran()

  result <- get_japan_child_mortality()

  # Number of rows (2010–2022 inclusive)
  expect_equal(nrow(result), 13)

  # Exact years expected
  expect_equal(sort(result$year), 2010:2022)

  # Should be sorted in descending order
  expect_true(all(diff(result$year) <= 0))
})

test_that("get_japan_child_mortality() returns consistent values for Japan", {
  skip_on_cran()

  result <- get_japan_child_mortality()

  # Country
  expect_true(all(result$country == "Japan"))

  # Indicator should match mortality
  expect_true(all(grepl("Mortality", result$indicator, ignore.case = TRUE)))
  expect_true(all(grepl("under-5", result$indicator, ignore.case = TRUE)))

  # Values must be positive and realistic (Japan < 10)
  expect_true(all(result$value > 0, na.rm = TRUE))
  expect_true(all(result$value < 10, na.rm = TRUE))
})

test_that("get_japan_child_mortality() values are numeric and non-missing", {
  skip_on_cran()

  result <- get_japan_child_mortality()

  # No missing years
  expect_true(all(!is.na(result$year)))

  # Value should be numeric and not NA
  expect_true(is.numeric(result$value))
  expect_true(all(!is.na(result$value)))
})

test_that("get_japan_child_mortality() handles API errors gracefully", {
  skip_on_cran()

  result <- tryCatch(get_japan_child_mortality(), error = function(e) NULL)

  # Return NULL or tibble
  expect_true(is.null(result) || inherits(result, "tbl_df"))
})
