#'Estimates levels of non-attention.
#'@description Functions to estimate E.S., income gain (I.G.), percentage of
#'  I.G.=P.I.G., and non-attention level (N.A.L.) of each S.S. per L.S., and
#'  their partial sum of I.G. and P.I.G. of S.S. inside each L.S., and the total
#'  of I.G. and P.I.G. on the productive system.
#'@usage
#'NonAttentionLevel(EffectivenessOfSolution,
#'LossProduction,
#'Id,
#'SafetyMargin=1.25,
#'Verbose=TRUE)
#'
#'@param EffectivenessOfSolution Output generated by the function
#'  `EffectivenessOfSolution`
#'@param LossProduction Output generated by the function  `LossProduction`
#'@param Id Logical vector indicating the lines of the `EffectivenessOfSolution`
#'  that are relevant. Output generated by the function
#'  SelectEffectivenessOfSolution
#'@param SafetyMargin Safety Margin (Default=1.25)
#'@param Verbose Logical value (TRUE/FALSE). TRUE displays the results of the
#'  analysis.
#'@author Germano Leao Demolin-Leite (Instituto de Ciencias Agrarias da UFMG) \cr
#' Alcinei Mistico Azevedo (Instituto de Ciencias Agrarias da UFMG)
#'@return The function returns levels of non-attention.
#'@seealso  \code{\link{EffectivenessOfSolution}} ,
#'  \code{\link{NonAttentionLevel}}  ,  \code{\link{LossSource}}
#'@importFrom stats lm
#'@export
#'@examples
#\dontrun{
#'library(ImportanceIndice)
#'data("DataLossSource")
#'data("DataSolutionSource")
#'data("DataProduction")
#'data("DataNumberSamples")
#'
#'Distribution_LossSource(DataLossSource)
#'Distribution_SolutionSource(DataSolutionSource)
#'
#'#################################################
#'###################################################
#'
#'
#'
#'LS<-LossSource(DataLoss = DataLossSource,DataProd = DataProduction)
#'LS
#'
#'LP<-LossProduction(Data=DataLossSource,Prod = DataProduction,
#'                  Evaluation=DataNumberSamples,
#'                  SegurityMargen=0.75,MaximumToleranceOfLossFruits=1)
#'LP
#'
#'
#'ES<-EffectivenessOfSolution(DataLossSource=DataLossSource,
#'                           DataSolutionSource=DataSolutionSource,Production =DataProduction)
#'ES
#'
#'
#'
#'
#'id<-SelectEffectivenessOfSolution(ES)
#'id<-c(TRUE , TRUE,  TRUE , FALSE,  TRUE)
#'
#'
#'SS<-SolutionSource(SolutionData = DataSolutionSource,
#'                  EffectivenessOfSolution = ES,Production = DataProduction,Id = id)
#'SS
#'
#'
#'NAL<-NonAttentionLevel(EffectivenessOfSolution = ES,LossProduction = LP,Id = id,Verbose=TRUE)
#'NAL
#}



NonAttentionLevel=function(EffectivenessOfSolution,LossProduction,Id,SafetyMargin=1.25,Verbose=TRUE){
  Es=EffectivenessOfSolution
  LS=LP=LossProduction$Res1

  Esb=Es[Id,]

  NAL=NULL
  for(i in 1:nrow(Esb)){
    id=rownames(LP)==Esb[i,2]
    I.G.=LP[id,1]*as.numeric(Esb[i,3])
    P.I.G.=100*I.G./as.numeric(LP[id,2])

    id2=rownames(LS)==Esb[i,2]

    id3=(Es[,1]==Esb[i,1])&(Es[,2]==Esb[i,2])

    AL=as.numeric(LP[id,5])
    Ess=as.numeric(Es[id3,3])
    NAL=rbind(NAL,c(Es=Ess,I.G.=I.G.,P.I.G.=P.I.G.,N.A.L.=(AL*(1*SafetyMargin))/Ess))

  }

  RES=cbind(Esb[,-3],NAL)

NAL=list(RES=cbind(Esb[,-3],NAL),Sum=colSums(RES[,-c(1:2,6)])[-1])

  if(Verbose==TRUE){
    un=unique(c(NAL$RES[,2]))

    cat(green("######################################################## \n"))
    cat(green("Increase in production per solution source \n"))
    cat(green("######################################################## \n"))
    for(i in un){
      iid=NAL$RES[,2]==i
      cat(green("________________________________________________________ \n"))
      cat(green(i,"\n"))
      cat(green("________________________________________________________ \n"))
      DDD=NAL$RES[iid,-2]
      row.names(DDD)=NULL
      print(DDD)
      cat("\n")
      print(paste("Sum: ", colSums(DDD[,3:4])))

    }


    cat(green("######################################################## \n"))
    cat(green("Increase in production total \n"))
    cat(green("######################################################## \n"))
    row.names(NAL)=NULL
    print(NAL)
  }




  (NAL)

}
