\name{plotHMMParameters}
\alias{plotHMMParameters}
\title{Plot Hidden Markov Model Parameters Over Time}
\description{
Plots the estimated state-dependent parameters of a fitted Hidden Markov Model (HMM) over time.
Confidence intervals can be included, along with optional overlay of observed data for visual comparison.
}
\usage{
plotHMMParameters(x, HMM, obsdist, confint_result = NULL,
                  level = 0.95, B = 100,
                  time_structure = NULL,
                  plot_title = "HMM Parameters Over Time",
                  overlay_data = NULL, overlay_label = "Data",
                  colors = c("black", "red", "blue", "green"),
                  save_plot = FALSE, filename = NULL,
                  width = 12, height = 8, dpi = 300,
                  verbose = TRUE, seed=NULL)
}
\arguments{
\item{x}{Numeric vector. The observed data sequence.}

\item{HMM}{A fitted HMM object (typically the output from \code{\link{findmleHMM}}),
containing estimated transition probabilities \code{Pi}, initial probabilities \code{delta},
and state-dependent observation parameters.}

\item{obsdist}{Character string. Observation distribution. Supported distributions include:
"norm", "pois", "weibull", "zip", "nbinom", "zinb", "exp", "gamma", "lnorm", "gev", "ZInormal", "ZIgamma".}

\item{confint_result}{Optional result from \code{\link{confintHMM}}.
If not provided, confidence intervals are computed internally.}

\item{level}{Numeric between 0 and 1. Confidence level for parameter intervals. Default is 0.95.}

\item{B}{Integer. Number of bootstrap replicates used when computing confidence intervals (if not supplied). Default is 100.}

\item{time_structure}{Optional list specifying the time scale for plotting.
May include \code{unit}, \code{observations_per_unit}, and optional \code{start_point}, \code{end_point}.
Supports calendar units (e.g., "day", "year") or custom scaling.}

\item{plot_title}{Character string. Title for the full parameter plot layout. Default is "HMM Parameters Over Time".}

\item{overlay_data}{Optional numeric vector of length \code{length(x)}. External series to overlay on parameter plots (e.g., covariates or raw data).}

\item{overlay_label}{Character string. Label for overlayed data axis. Default is "Data".}

\item{colors}{Character vector of plotting colors. Default is \code{c("black", "red", "blue", "green")}.}

\item{save_plot}{Logical. If \code{TRUE}, saves the plot to file. Default is \code{FALSE}.}

\item{filename}{Character string or \code{NULL}. Filename for saved plot (if \code{save_plot = TRUE}).
Must be specified when \code{save_plot = TRUE}. Default is \code{NULL}.}

\item{width}{Numeric. Width of saved plot in inches. Default is 12.}

\item{height}{Numeric. Height of saved plot in inches. Default is 8.}

\item{dpi}{Integer. Resolution of saved plot in dots per inch. Default is 300.}

\item{verbose}{Logical. If \code{TRUE}, progress messages are printed to the console. Default is \code{TRUE}.}

\item{seed}{Integer or \code{NULL}. Random seed for reproducibility. Default is \code{NULL}.}
}
\value{
Invisibly returns a list containing:
\item{param_series}{List of parameter time series for each parameter.}
\item{ci_series}{List of confidence interval time series (lower and upper bounds) for each parameter.}
\item{time_info}{List containing time axis information and labels.}
\item{decoded_states}{Vector of decoded hidden states for each time point.}

The function also produces parameter trajectory plots with confidence intervals and optional overlays.
}

\details{
For each observation, the most likely hidden state is decoded using \code{\link{globaldecodeHMM}}.
The corresponding state-dependent parameters are extracted and plotted over time.
If \code{confint_result} is not supplied, bootstrap confidence intervals are computed via \code{\link{confintHMM}}.
Custom time scaling can be applied using the \code{time_structure} argument.
Overlay data (such as covariates or raw series) can be plotted on a secondary axis for comparison.
When \code{verbose = TRUE}, progress messages are displayed during confidence interval computation.
}
\examples{
set.seed(123)
J <- 2
Pi <- matrix(c(0.9, 0.1,
               0.2, 0.8), nrow = 2, byrow = TRUE)
delta <- c(0.5, 0.5)
obspar <- list(lambda = c(2, 7))

sim_data <- generateHMM(n = 120, J = J, obsdist = "pois",
                        obspar = obspar, Pi = Pi, delta = delta)

HMM_fit <- findmleHMM(x = sim_data$x, J = J, obsdist = "pois",
                      obspar = obspar, Pi = Pi, delta = delta)

overlay_series <- sim_data$x

time_struct <- list(unit = "day", observations_per_unit = 12, start_point = 1)

\donttest{
result <- plotHMMParameters(x = sim_data$x, HMM = HMM_fit, obsdist = "pois",
                            overlay_data = overlay_series,
                            overlay_label = "Observed counts",
                            time_structure = time_struct,
                            plot_title = "Poisson HMM Parameters with Overlay")

result <- plotHMMParameters(x = sim_data$x, HMM = HMM_fit, obsdist = "pois",
                            overlay_data = overlay_series,
                            overlay_label = "Observed counts",
                            time_structure = time_struct,
                            save_plot = TRUE,
                            filename = tempfile(fileext = ".png"),
                            verbose = FALSE)
}
}
\seealso{
\code{\link{findmleHMM}}, for fitting HMMs.
\code{\link{generateHMM}}, for simulating HMM data.
\code{\link{globaldecodeHMM}}, for decoding the most likely state sequence.
\code{\link{confintHMM}}, for bootstrap confidence intervals of HMM parameters.
}
\author{Aimee Cody}
