#' Plot GACE Forecast
#'
#' Produces a plot of historical and forecast values returned by 
#' gace_forecast(). Includes stability handling for missing values,
#' non-numeric periods, and clean ggplot2 output.
#'
#' @param fc A data frame returned by gace_forecast(), containing:
#'   \itemize{
#'     \item \code{period} – numeric or convertible index,
#'     \item \code{value}  – observed or forecast values,
#'     \item \code{type}   – "historical" or "forecast".
#'   }
#'
#' @return A ggplot2 object.
#'
#' @examples
#' \donttest{
#'   set.seed(1)
#'   y <- ts(rnorm(48, mean = 100, sd = 10), frequency = 12)
#'   fc <- gace_forecast(y, periods = 6, freq = "month")
#'   plot_gace(fc)
#' }
#'
#' @export
plot_gace <- function(fc) {
  
  # Validate required columns
  required_cols <- c("period", "value", "type")
  if (!all(required_cols %in% names(fc))) {
    stop("`fc` must contain: period, value, type. Use gace_forecast() to generate it.")
  }
  
  # Convert period safely
  if (!is.numeric(fc$period)) {
    if (inherits(fc$period, "Date") || inherits(fc$period, "POSIXct")) {
      fc$period <- as.numeric(fc$period)
    } else {
      fc$period <- as.numeric(as.factor(fc$period))
    }
  }
  
  # Clean values
  fc$value[!is.finite(fc$value)] <- NA_real_
  
  if (all(is.na(fc$value[fc$type == "forecast"]))) {
    warning("GACE forecast contains all NA values; plotting history only.")
  }
  
  # Normalize type
  fc$type <- as.character(fc$type)
  valid_types <- c("historical", "forecast")
  fc$type[!(fc$type %in% valid_types)] <- "historical"
  
  # Plot
  ggplot2::ggplot(fc, ggplot2::aes(x = period, y = value, color = type)) +
    ggplot2::geom_line(linewidth = 1.2, na.rm = TRUE) +
    ggplot2::geom_point(size = 2.2, na.rm = TRUE) +
    ggplot2::scale_color_manual(
      values = c("historical" = "#2c7bb6",
                 "forecast"   = "#d7191c")
    ) +
    ggplot2::theme_minimal(base_size = 13) +
    ggplot2::theme(
      legend.position = "bottom",
      legend.title    = ggplot2::element_blank(),
      plot.title      = ggplot2::element_text(hjust = 0.5, face = "bold"),
      axis.title      = ggplot2::element_text(face = "bold")
    ) +
    ggplot2::labs(
      title    = "GACE Forecast",
      subtitle = "Historical vs Forecast",
      x        = "Period",
      y        = "Value"
    )
}