% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EM.skewCens.mixR.R
\name{EM.skewCens.mixR}
\alias{EM.skewCens.mixR}
\title{EM Algorithm for Finite Mixture Censored Regression}
\usage{
EM.skewCens.mixR(
  cc,
  y,
  x,
  Abetas = NULL,
  sigma2 = NULL,
  shape = NULL,
  pii = NULL,
  nu = NULL,
  g = NULL,
  get.init = TRUE,
  criteria = TRUE,
  group = FALSE,
  family = "Normal",
  error = 1e-05,
  iter.max = 100,
  obs.prob = FALSE,
  kmeans.param = NULL,
  aitken = TRUE,
  IM = TRUE
)
}
\arguments{
\item{cc}{Integer vector of length \code{n}; censoring indicator (1 = censored, 0 = observed).}

\item{y}{Numeric response vector (univariate).}

\item{x}{Numeric design matrix (\code{n x p}); include intercept column if needed.}

\item{Abetas}{Optional initial regression coefficient matrix (\code{p x g}).}

\item{sigma2}{Optional initial variance(s), length \code{g}.}

\item{shape}{Optional initial skewness parameter(s), length \code{g} (used in SN/ST).}

\item{pii}{Optional initial mixing proportions, length \code{g}, must sum to 1.}

\item{nu}{Degrees of freedom for T/ST models (scalar).}

\item{g}{Number of mixture components (\eqn{g \ge 1}). Required if \code{get.init = TRUE}.}

\item{get.init}{Logical; if \code{TRUE}, k-means-based initialization is used.}

\item{criteria}{Logical; if \code{TRUE}, returns AIC/BIC/EDC.}

\item{group}{Logical; if \code{TRUE}, returns hard cluster labels.}

\item{family}{One of \code{"Normal"}, \code{"T"}, \code{"SN"}, \code{"ST"}.}

\item{error}{Convergence tolerance for EM.}

\item{iter.max}{Maximum number of EM iterations.}

\item{obs.prob}{Logical; if \code{TRUE}, returns posterior membership matrix.}

\item{kmeans.param}{Optional list for kmeans init.}

\item{aitken}{Logical; use Aitken acceleration for convergence monitoring.}

\item{IM}{Logical; if \code{TRUE}, compute (robust) standard errors via information matrix.}
}
\value{
A list with elements:
\item{Abetas}{Estimated regression coefficients (\code{p x g}).}
\item{sigma2}{Estimated variances (length \code{g}).}
\item{shape}{Estimated skewness parameters (length \code{g}; SN/ST).}
\item{pii}{Estimated mixing proportions (length \code{g}).}
\item{sd}{Standard errors (if \code{IM=TRUE}).}
\item{nu}{Estimated/used degrees of freedom (T/ST).}
\item{loglik}{Final log-likelihood.}
\item{loglikT}{Log-likelihood trace over iterations.}
\item{aic,bic,edc}{Information criteria (if \code{criteria=TRUE}).}
\item{iter}{Number of EM iterations.}
\item{n}{Sample size.}
\item{group}{Hard labels (if \code{group=TRUE}).}
}
\description{
Fits finite mixture censored regression models under four families:
Normal ("Normal"), Student-t ("T"), Skew-Normal ("SN"), and Skew-t ("ST").
}
\details{
Left-censoring is indicated by \code{cc[i] = 1} and replacing \code{y[i]} by the censoring point.
The routine supports Normal, t, Skew-Normal, and Skew-t families with finite mixtures.
}
\examples{
\dontshow{if (requireNamespace("mixsmsn", quietly = TRUE)) withAutoprint(\{ # examplesIf}
set.seed(1)
n <- 150
X <- cbind(1, runif(n), rnorm(n))
pi <- c(0.6, 0.4); nu <- 4
b1 <- c(0.5, 1.0, -1.0); sigma1 <- 1; shape1 <- 2
b2 <- c(1.0,-0.5, 0.5);  sigma2 <- 2; shape2 <- 3
mu1 <- drop(X \%*\% b1); mu2 <- drop(X \%*\% b2)
draw1 <- function(i){
  a1 <- list(mu=mu1[i], sigma2=sigma1, shape=shape1, nu=nu)
  a2 <- list(mu=mu2[i], sigma2=sigma2, shape=shape2, nu=nu)
  mixsmsn::rmix(1, pi, "Skew.t", list(a1,a2), cluster=FALSE)
}
y0 <- vapply(seq_len(n), draw1, numeric(1))
cutoff <- unname(stats::quantile(y0, 0.20))
cc <- as.integer(y0 <= cutoff)
y  <- ifelse(cc == 1, cutoff, y0)
fit <- EM.skewCens.mixR(cc, y, X, g=2, family="Normal", iter.max=50)
fit$loglik
\dontshow{\}) # examplesIf}
}
