#' Prepare and render the HTML5 report
#'
#' This script generates the HTML5 report based on an enriched
#' `CALANGO`-type list output by `run_CALANGO()`.
#'
#' @param defs an enriched `CALANGO`-type list generated by `run_CALANGO()`
#' @param render.report logical: should the HTML5 report be generated (for
#' internal use only)
#'
#' @return Updated `defs` list, containing:
#' \itemize{
#'    \item All input parameters originally passed (see [run_CALANGO()] for 
#'    details).
#'    \item Derived fields calculated for displaying the results, including 
#'    several statistical summaries of the data (including correlations, 
#'    contrasts, covariances, p-values).
#' }
#' 
#' This function is mainly called for its side effect, namely the generation of 
#' an HTML5 report of the analysis, which is saved to the folder indicated in 
#' `defs$output.dir`.
#'
#' @export

make_report <- function(defs, render.report = TRUE){

  # ================== Sanity checks ==================
  assertthat::assert_that(is.logical(render.report),
                          length(render.report) == 1,
                          all(c("CALANGO", "list") %in% class(defs)))

  if(render.report){
    defs <- switch(tolower(defs$type),
                   significance = make_significance_report(defs),
                   correlation  = make_correlation_report(defs))

    # Open in browser
    utils::browseURL(gsub("//", "/", paste0(normalizePath(defs$output.dir),
                                            "/CALANGO_report.html"),
                          fixed = TRUE))
  }

  invisible(defs)
}
