% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{bws_cdf}
\alias{bws_cdf}
\title{CDF of the Baumgartner-Weiss-Schindler test under the null.}
\usage{
bws_cdf(b, maxj = 5L, lower_tail = TRUE)
}
\arguments{
\item{b}{a vector of BWS test statistics.}

\item{maxj}{the maximum value of j to take in the approximate computation
of the CDF via equation (2.5). Baumgartner \emph{et. al.} claim that a
value of 3 is sufficient.}

\item{lower_tail}{boolean, when \code{TRUE} returns \eqn{\Psi}{Psi}, otherwise
compute the upper tail, \eqn{1-\Psi}{1 - Psi}, which is more useful for hypothesis tests.}
}
\value{
A vector of the CDF of \eqn{b}, \eqn{\Psi(b)}{Psi(b)}.
}
\description{
Computes the CDF of the Baumgartner-Weiss-Schindler test statistic under the
null hypothesis of equal distributions.
}
\details{
Given value \eqn{b}, computes the CDF of the BWS statistic under
the null, denoted as \eqn{\Psi(b)}{Psi(b)} by 
Baumgartner \emph{et al.}  The CDF is computed from 
equation (2.5) via numerical quadrature.

The expression for the CDF contains the integral
\deqn{\int_0^1 \frac{1}{\sqrt{r^3 (1-r)}} \mathrm{exp}\left(\frac{rb}{8} - \frac{\pi^2 (4j+1)^2}{8rb}\right) \mathrm{dr}}
By making the change of variables \eqn{x = 2r - 1}, this can
be re-expressed as an integral of the form
\deqn{\int_{-1}^1 \frac{1}{\sqrt{1-x^2}} f(x) \mathrm{dx},}
for some function \eqn{f(x)} involving \eqn{b} and \eqn{j}. 
This integral can be approximated
via Gaussian quadrature using Chebyshev nodes (of the first kind), which
is the approach we take here.
}
\examples{

# do it 500 times
set.seed(123)
bvals <- replicate(500, bws_stat(rnorm(50),rnorm(50)))
pvals <- bws_cdf(bvals)
# these should be uniform!
\donttest{ 
  plot(ecdf(pvals)) 
}

# compare to Table 1 of Baumgartner et al.
bvals <- c(1.933,2.493,3.076,3.880,4.500,5.990)
tab1v <- c(0.9,0.95,0.975,0.990,0.995,0.999)
pvals <- bws_cdf(bvals,lower_tail=TRUE)
show(data.frame(B=bvals,BWS_psi=tab1v,our_psi=pvals))

}
\references{
W. Baumgartner, P. Weiss, H. Schindler, 'A nonparametric test for the general two-sample problem', 
Biometrics 54, no. 3 (Sep., 1998): pp. 1129-1135. 
\doi{10.2307/2533862}
}
\seealso{
\code{\link{bws_stat}}, \code{\link{bws_test}}
}
\author{
Steven E. Pav \email{shabbychef@gmail.com}
}
