#' Visual Studio Project Git Repositories
#'
#' @description
#' These functions will allow you to scrape git repository information from Visual Studio.
#'
#' @details
#' For more information about git repository API calls check
#' \url{https://docs.microsoft.com/en-us/rest/api/vsts/git/}.
#'
#' @param domain the location of the visual studio server
#' @param project the name of the project in \code{domain} to look at
#' @param repo the name of the repository in \code{project} to look at
#' @param auth_key authentication key generated by using \code{\link{vsts_auth_key}}
#' @param query a list of extra parameters that can be sent to the API call:
#' \describe{
#' \item{\code{branch}}{[character] the name of a branch in the repository (cannot combine with \code{commit})}
#' \item{\code{commit}}{[character] the id of a commit in the repository (cannot combine with \code{branch})}
#' \item{\code{itemPath}}{[character] path of an item in the repository}
#' \item{\code{committer}}{[character] name of the person who committed the change}
#' \item{\code{author}}{[character] name of the author}
#' \item{\code{fromDate}}{[Date] start date to search from}
#' \item{\code{toDate}}{[Date] end date to search from}
#' }
#'
#' @examples
#' #Add in own details to get a non-NULL output
#' auth_key <- vsts_auth_key('<username>', '<password>')
#' vsts_get_commits('domain', 'project', 'repo', auth_key)
#'
#' @export
vsts_get_commits <- function(domain, project, repo, auth_key, query = NULL) {
  repo_id <- vsts_get_repos(domain, project, auth_key, quiet = TRUE) %>% .[.$name == repo, 'id']
  if(length(repo_id) == 0) {
    cat('Unable to find ', repo, ' in ', project, '.\n', sep = '')
    return(invisible(NULL))
  }
  uri <- paste0('https://', domain, '.visualstudio.com/DefaultCollection/', project, '/_apis/git/repositories/', repo_id,
                '/commits?api-version=1.0')

  response <- httr::GET(uri, httr::add_headers(Authorization = auth_key), query = query)
  if(httr::status_code(response) != 200) {
    cat(httr::http_condition(response, 'message', 'get commit list')$message, '\n')
    return(invisible(NULL))
  }

  content <- httr::content(response, as = 'text', encoding = 'UTF-8') %>% jsonlite::fromJSON(., flatten = TRUE) %>% .$value
  return(invisible(content))
}
