% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/use_cassette.R
\name{use_cassette}
\alias{use_cassette}
\title{Use a cassette to record HTTP requests}
\usage{
use_cassette(
  name,
  ...,
  record = NULL,
  match_requests_on = NULL,
  update_content_length_header = FALSE,
  allow_playback_repeats = FALSE,
  serialize_with = NULL,
  persist_with = NULL,
  preserve_exact_body_bytes = NULL,
  re_record_interval = NULL,
  clean_outdated_http_interactions = NULL
)
}
\arguments{
\item{name}{The name of the cassette. vcr will check this to ensure it
is a valid file name. Not allowed: spaces, file extensions, control
characters (e.g., \verb{\\n}), illegal characters ('/', '?', '<', '>', '\\', ':',
'*', '|', and '\"'), dots alone (e.g., '.', '..'), Windows reserved
words (e.g., 'com1'), trailing dots (can cause problems on Windows),
names longer than 255 characters. See section "Cassette names"}

\item{...}{a block of code containing one or more requests (required). Use
curly braces to encapsulate multi-line code blocks. If you can't pass a code
block use \code{\link[=insert_cassette]{insert_cassette()}} instead.}

\item{record}{The record mode (default: \code{"once"}). See \link{recording} for a
complete list of the different recording modes.}

\item{match_requests_on}{List of request matchers
to use to determine what recorded HTTP interaction to replay. Defaults to
\verb{["method", "uri"]}. The built-in matchers are "method", "uri", "host",
"path", "headers", "body" and "query"}

\item{update_content_length_header}{(logical) Whether or
not to overwrite the \code{Content-Length} header of the responses to
match the length of the response body. Default: \code{FALSE}}

\item{allow_playback_repeats}{(logical) Whether or not to
allow a single HTTP interaction to be played back multiple times.
Default: \code{FALSE}.}

\item{serialize_with}{(character) Which serializer to use.
Valid values are "yaml" (default), the only one supported for now.}

\item{persist_with}{(character) Which cassette persister to
use. Default: "file_system". You can also register and use a
custom persister.}

\item{preserve_exact_body_bytes}{(logical) Whether or not
to base64 encode the bytes of the requests and responses for
this cassette when serializing it. See also \code{preserve_exact_body_bytes}
in \code{\link[=vcr_configure]{vcr_configure()}}. Default: \code{FALSE}}

\item{re_record_interval}{(integer) How frequently (in seconds) the
cassette should be re-recorded. default: \code{NULL} (not re-recorded)}

\item{clean_outdated_http_interactions}{(logical) Should outdated
interactions be recorded back to file? default: \code{FALSE}}
}
\value{
an object of class \code{Cassette}
}
\description{
Use a cassette to record HTTP requests
}
\details{
A run down of the family of top level \pkg{vcr} functions
\itemize{
\item \code{use_cassette} Initializes a cassette. Returns the inserted
cassette.
\item \code{insert_cassette} Internally used within \code{use_cassette}
\item \code{eject_cassette} ejects the current cassette. The cassette
will no longer be used. In addition, any newly recorded HTTP interactions
will be written to disk.
}
}
\section{Cassette options}{


Default values for arguments controlling cassette behavior are
inherited from vcr's global configuration. See \code{\link[=vcr_configure]{vcr_configure()}} for a
complete list of options and their default settings. You can override these
options for a specific cassette by changing an argument's value to something
other than \code{NULL} when calling either \code{insert_cassette()} or
\code{use_cassette()}.
}

\section{Behavior}{

This function handles a few different scenarios:
\itemize{
\item when everything runs smoothly, and we return a \code{Cassette} class object
so you can inspect the cassette, and the cassette is ejected
\item when there is an invalid parameter input on cassette creation,
we fail with a useful message, we don't return a cassette, and the
cassette is ejected
\item when there is an error in calling your passed in code block,
we return with a useful message, and since we use \code{on.exit()}
the cassette is still ejected even though there was an error,
but you don't get an object back
}
}

\section{Cassettes on disk}{

Note that \emph{"eject"} only means that the R session cassette is no longer
in use. If any interactions were recorded to disk, then there is a file
on disk with those interactions.
}

\section{Using with tests (specifically \pkg{testthat})}{

There's a few ways to get correct line numbers for failed tests and
one way to not get correct line numbers:

\emph{Correct}: Either wrap your \code{test_that()} block inside your \code{use_cassette()}
block, OR if you put your \code{use_cassette()} block inside your \code{test_that()}
block put your \code{testthat} expectations outside of the \code{use_cassette()}
block.

\emph{Incorrect}: By wrapping the \code{use_cassette()} block inside your
\code{test_that()} block with your \pkg{testthat} expectations inside the
\code{use_cassette()} block, you'll only get the line number that the
\code{use_cassette()} block starts on.
}

\section{Cassette names}{

\itemize{
\item Should be meaningful so that it's obvious to you what test/function
they relate to. Meaningful names are important so that you can quickly
determine to what test file or test block a cassette belongs. Note that
vcr cannot check that your cassette names are meaningful.
\item Should not be duplicated. Duplicated cassette names would lead to
a test using the wrong cassette.
\item Should not have spaces. Spaces can lead to problems in using file paths.
\item Should not include a file extension. vcr handles file extensions for
the user.
\item Should not have illegal characters that can lead to problems in using
file paths: '/', '?', '<', '>', '\\', ':', '*', '|', and '\"'
\item Should not have control characters, e.g., \verb{\\n}
\item Should not have just dots, e.g,. \code{.} or \code{..}
\item Should not have Windows reserved words, e.g, \code{com1}
\item Should not have trailing dots
\item Should not be longer than 255 characters
}
}

\examples{
\dontrun{
library(vcr)
library(crul)
vcr_configure(dir = tempdir())

use_cassette(name = "apple7", {
  cli <- HttpClient$new(url = "https://httpbin.org")
  resp <- cli$get("get")
})
readLines(file.path(tempdir(), "apple7.yml"))

# preserve exact body bytes - records in base64 encoding
use_cassette("things4", {
  cli <- crul::HttpClient$new(url = "https://httpbin.org")
  bbb <- cli$get("get")
}, preserve_exact_body_bytes = TRUE)
## see the body string value in the output here
readLines(file.path(tempdir(), "things4.yml"))

# cleanup
unlink(file.path(tempdir(), c("things4.yml", "apple7.yml")))


# with httr
library(vcr)
library(httr)
vcr_configure(dir = tempdir(), log = TRUE)

use_cassette(name = "stuff350", {
  res <- GET("https://httpbin.org/get")
})
readLines(file.path(tempdir(), "stuff350.yml"))

use_cassette(name = "catfact456", {
  res <- GET("https://catfact.ninja/fact")
})

# record mode: none
library(crul)
vcr_configure(dir = tempdir())

## make a connection first
conn <- crul::HttpClient$new("https://eu.httpbin.org")
## this errors because 'none' disallows any new requests
# use_cassette("none_eg", (res2 <- conn$get("get")), record = "none")
## first use record mode 'once' to record to a cassette
one <- use_cassette("none_eg", (res <- conn$get("get")), record = "once")
one; res
## then use record mode 'none' to see it's behavior
two <- use_cassette("none_eg", (res2 <- conn$get("get")), record = "none")
two; res2
}
}
\seealso{
\code{\link[=insert_cassette]{insert_cassette()}}, \code{\link[=eject_cassette]{eject_cassette()}}
}
