% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SVC_mle.R
\name{SVC_mle}
\alias{SVC_mle}
\alias{SVC_mle.default}
\alias{SVC_mle.formula}
\title{MLE of SVC model}
\usage{
SVC_mle(...)

\method{SVC_mle}{default}(y, X, locs, W = NULL, control = NULL, optim.control = list(), ...)

\method{SVC_mle}{formula}(
  formula,
  data,
  RE_formula = NULL,
  locs,
  control = NULL,
  optim.control = list(),
  ...
)
}
\arguments{
\item{...}{further arguments}

\item{y}{(\code{numeric(n)}) \cr
Response vector.}

\item{X}{(\code{matrix(n, p)}) \cr
Design matrix. Intercept has to be added manually.}

\item{locs}{(\code{matrix(n, d)}) \cr
Locations in a \eqn{d}-dimensional space. May contain multiple
observations at single location.}

\item{W}{(\code{NULL} or \code{matrix(n, q)}) \cr
If \code{NULL}, the same matrix as provided in \code{X} is used. This
fits a full SVC model, i.e., each covariate effect is modeled with a mean
and an SVC. In this case we have \eqn{p = q}. If optional matrix \code{W}
is provided, SVCs are only modeled for covariates within matrix \code{W}.}

\item{control}{(\code{list}) \cr
Control paramaters given by \code{\link{SVC_mle_control}}.}

\item{optim.control}{(\code{list}) \cr
Control arguments for optimization function, see Details in
\code{\link{optim}}.}

\item{formula}{Formula describing the fixed effects in SVC model. The response, 
i.e. LHS of the formula, is not allowed to have functions such as \code{sqrt()} or \code{log()}.}

\item{data}{data frame containing the observations}

\item{RE_formula}{Formula describing the random effects in SVC model. 
Only RHS is considered. If \code{NULL}, the same RHS of argument \code{formula} for fixed effects is used.}
}
\value{
Object of class \code{SVC_mle} if \code{control$extract_fun = FALSE},
meaning that a MLE has been conducted. Otherwise, if \code{control$extract_fun = TRUE},
the function returns a list with two entries:
\itemize{
   \item \code{obj_fun}: the objective function used in the optimization
   \item \code{args}: the arguments to evaluate the objective function.
}
For further details, see description of \code{\link{SVC_mle_control}}.
}
\description{
Conducts a maximum likelihood estimation (MLE) for a Gaussian
  process-based spatially varying coefficient model as described in 
  Dambon et al. (2021) \doi{10.1016/j.spasta.2020.100470}.
}
\details{
The GP-based SVC model is defined with some abuse of notation as:

\deqn{y(s) = X \mu + W \eta (s) + \epsilon(s)}

where:
\itemize{
  \item \eqn{y} is the response (vector of length \eqn{n})
  \item \eqn{X} is the data matrix for the fixed effects covariates. The
  dimensions are \eqn{n} times \eqn{p}. This leads to \eqn{p} fixed effects.
  \item \eqn{\mu} is the vector containing the fixed effects
  \item W is the data matrix for the SVCs modeled by GPs. The dimensions are
  \eqn{n} times \eqn{q}. This lead to \eqn{q} SVCs in the model.
  \item \eqn{\eta} are the SVCs represented by a GP.
  \item \eqn{\epsilon} is the nugget effect
}

The MLE is an numeric optimization that runs \code{\link[stats]{optim}} or
(if parallelized) \code{\link[optimParallel]{optimParallel}}.

You can call the function in two ways. Either, you define the model matrices
yourself and provide them using the arguments \code{X} and \code{W}. As usual, 
the individual columns correspond to the fixed and random effects, i.e., the 
Gaussian processes, respectively. The second way is to call the function with
formulas, like you would in \code{\link[stats]{lm}}. From the \code{data.frame}
provided in argument \code{data}, the respective model matrices as described
above are implicitly built. Using simple arguments \code{formula} and 
\code{RE_formula} with \code{data} column names, we can decide which 
covariate is modeled with a fixed or random effect (SVC). 

Note that similar to model matrix call from above, if the \code{RE_formula} 
is not provided, we use the one as in argument \code{formula}. Further, note 
that the intercept is implicitly constructed in the model matrix if not 
prohibited.
}
\examples{
## ---- toy example ----
## We use the sampled, i.e., one dimensional SVCs
str(SVCdata)
# sub-sample data to have feasible run time for example
set.seed(123)
id <- sample(length(SVCdata$locs), 50)

## SVC_mle call with matrix arguments
fit <- with(SVCdata, SVC_mle(
  y[id], X[id, ], locs[id], 
  control = SVC_mle_control(profileLik = TRUE, cov.name = "mat32")))

## SVC_mle call with formula
df <- with(SVCdata, data.frame(y = y[id], X = X[id, -1]))
fit <- SVC_mle(
  y ~ X, data = df, locs = SVCdata$locs[id], 
  control = SVC_mle_control(profileLik = TRUE, cov.name = "mat32")
)
class(fit)

summary(fit)

\donttest{
## ---- real data example ----
require(sp)
## get data set
data("meuse", package = "sp")

# construct data matrix and response, scale locations
y <- log(meuse$cadmium)
X <- model.matrix(~1+dist+lime+elev, data = meuse)
locs <- as.matrix(meuse[, 1:2])/1000


## starting MLE
# the next call takes a couple of seconds
fit <- SVC_mle(
  y = y, X = X, locs = locs,
  # has 4 fixed effects, but only 3 random effects (SVC)
  # elev is missing in SVC
  W = X[, 1:3],
  control = SVC_mle_control(
    # inital values for 3 SVC
    # 7 = (3 * 2 covariance parameters + nugget)
    init = c(rep(c(0.4, 0.2), 3), 0.2),
    profileLik = TRUE
  )
)

## summary and residual output
summary(fit)
plot(fit)

## predict
# new locations
newlocs <- expand.grid(
  x = seq(min(locs[, 1]), max(locs[, 1]), length.out = 30),
  y = seq(min(locs[, 2]), max(locs[, 2]), length.out = 30))
# predict SVC for new locations
SVC <- predict(fit, newlocs = as.matrix(newlocs))
# visualization
sp.SVC <- SVC
coordinates(sp.SVC) <- ~loc_1+loc_2
spplot(sp.SVC, colorkey = TRUE)
}
}
\references{
Dambon, J. A., Sigrist, F., Furrer, R. (2021)
   \emph{Maximum likelihood estimation of spatially varying coefficient
   models for large data with an application to real estate price prediction},
   Spatial Statistics \doi{10.1016/j.spasta.2020.100470}
}
\seealso{
\code{\link{predict.SVC_mle}}
}
\author{
Jakob Dambon
}
