% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/varcast.R
\name{varcast}
\alias{varcast}
\title{Calculation of one-step ahead forecasts of Value at Risk and Expected Shortfall (parametric and
semiparametric)}
\usage{
varcast(
  x,
  a.v = 0.99,
  a.e = 0.975,
  model = c("sGARCH", "lGARCH", "eGARCH", "apARCH", "fiGARCH", "filGARCH"),
  garchOrder = c(1, 1),
  distr = c("norm", "std"),
  n.out = 250,
  smooth = c("none", "lpr"),
  ...
)
}
\arguments{
\item{x}{a vector containing the price series.}

\item{a.v}{confidence level for calculating VaR; is set to \code{0.99} by default.}

\item{a.e}{confidence level for calculating ES; is set to \code{0.975} by default.}

\item{model}{model to be estimated. Options are 'sGARCH', 'eGARCH', 'apARCH',
'lGARCH', 'fiGARCH' and 'filGARCH'; is set to \code{'sGARCH'} by default.}

\item{garchOrder}{orders to be estimated; c(1, 1), i.e. p = q = 1, is the
default.}

\item{distr}{distribution to use for the innovations of the respective GARCH model;
is set to \code{'std'} by default}

\item{n.out}{size of out-sample; is set to \code{250} by default.}

\item{smooth}{a character object; defines the data-driven smoothing approach
for the estimation of the nonparametric scale function; for
\code{smooth = 'lpr'}, the scale function is obtained from the logarithm
of the squared centralized returns by means of the \code{msmooth()} function or
\code{tsmoothlm()} function if \code{model} is set to \code{'sGARCH'}, \code{'eGARCH'},
\code{'apARCH'} and \code{lGARCH'} or \code{'fiGARCH} and \code{'filGARCH'},
respectively; is set to \code{smooth = 'none'} by default.}

\item{...}{depending on the choice of \code{model}, further arguments can be
passed to either \code{smoots::msmooth()} or to \code{tsmoothlm()};
if no other arguments are given, the default settings
are used for both functions with the exception of \code{p = 3}.}
}
\value{
This function returns a list with the following elements.
\describe{
\item{model}{selected model for estimation}
\item{mean}{the estimated mean of the in-sample returns}
\item{model.fit}{estimated model parameters for the parametric part of the
in-sample}
\item{np.est}{the estimation results for the nonparametric part of the
in-sample model}
\item{ret.in}{in-sample return series}
\item{ret.out}{out-sample return series}
\item{sig.in}{estimated in-sample total volatility}
\item{sig.fc}{out-sample forecasts of the total volatility}
\item{scale}{the estimated nonparametric scale function values for the
in-sample}
\item{scale.fc}{the scale function forecast for the out-sample}
\item{VaR.e}{out-sample forecasts of the (1-\code{a.e})100\% VaR}
\item{VaR.v}{out-sample forecasts of the (1-\code{a.v})100\% VaR}
\item{ES}{out-sample forecasts of the (1-\code{a.e})100\% ES}
\item{dfree}{estimated degrees of freedom for the standardized returns}
\item{a.v}{coverage level for the 99 \% VaR}
\item{a.e}{coverage level for 97.5 \% VaR}
\item{garchOrder}{the orders p and q of the implemented GARCH-type model}
}
}
\description{
One-step ahead forecasts of Value at Risk and Expected Shortfall for a selection of short-memory
and long-memory parametric as well as semiparametric GARCH-type models are
computed.
}
\details{
Let \eqn{Y_t} be a (demeaned) return series. The semiparametric extension of
the GARCH(p,q) model (Bollerslev, 1986) is called a Semi-GARCH model
(Feng, 2004) and is defined by

\deqn{Y_t = s(x_t)\sigma_t \eta_t,}
 with \eqn{\eta_t \sim IID(0,1)} and

\deqn{\sigma^2_t = \alpha_0 + \sum_{i=1}^p \alpha_i Y^2_{t-i}
+ \sum_{j=1}^q \beta_j \sigma^2_{t-j},}

where \eqn{\sigma_t > 0} are the conditional standard deviations, \eqn{s(x_t) > 0} is
a nonparametric scale function with \eqn{x_t} being the rescaled observation
time points on the interval [0,1] and \eqn{\alpha_i}
and \eqn{\beta_j} are non-negative real valued coefficients, except for
\eqn{\alpha_0}, which must satisfy \eqn{\alpha_0 > 0}. Furthermore, it is assumed that
Var\eqn{(\sigma_t \eta_t) = 1}. In this functions, different
short-memory and long-memory GARCH-type models are selectable for the parametric part of the
model. More specifically, the standard GARCH (Bollerslev, 1986), the
Log-GARCH (Pantula, 1986; Geweke, 1986; Milhoj, 1988), the eGARCH
(Nelson, 1991), the APARCH (Ding et al., 1993), the FIGARCH (Baillie et al., 1996)
and the FI-Log-GARCH (Feng et al., 2020) model are implemented. For more
information on the representations of the last three models mentioned, we
refer the reader to the corresponding references listed in the references
section.

While the innovations \eqn{\eta_t} must be i.i.d. (independent and identically
distributed) with zero-mean and unit-variance and while any
distribution that satisfies these conditions is suitable, the standardized
t-distribution is selected for the estimation of the models and computation
of the Value at Risk (VaR) as well as the Expected Shortfall (ES) within
this function.

For a given level \eqn{\alpha \in (0, 1)},

\deqn{VaR(\alpha) = inf \{z \in R: F_L(z) \geq \alpha\}}

defines the VaR at level alpha. In this definition, \eqn{L} is the
loss variable (making a loss is denoted as a positive value, whereas gains
are negative values) and \eqn{F_L} is its cumulative distribution function.
Explained differently, \eqn{VaR(\alpha)} is the \eqn{\alpha}-quantile of the loss
distribution.

The ES for a level \eqn{\alpha}, however, is given by

\deqn{ES(\alpha) = (1 / (1 - \alpha)) \int_{\alpha}^1 VaR(u)du,}

i.e. it is the expected loss in case \eqn{VaR(\alpha)} is exceeded. More
information on these risk measures can be found on pp. 64-72 in McNeil et
al. (2015).

To apply the function, a numeric vector that contains the price series that
is to be analyzed ordered from past to present must be passed to the
argument \code{x}. Furthermore, the user can set different levels of alpha
for the VaR and the ES via the arguments \code{a.v} and \code{a.e},
respectively. A parametric short-memory or long-memory GARCH-type model can be
selected by means of \code{model}, which only accepts a single-element character vector
as input. At the time of the release of package version 1.0.0, a standard
GARCH ('sGARCH'), a Log-GARCH ('lGARCH'), an eGARCH ('eGARCH'), an APARCH
('apARCH'), a FIGARCH ('fiGARCH') and a FI-Log-GARCH ('filGARCH') model can be selected,
each with conditional t-distribution. By default, a standard GARCH model is applied.
The orders of the GARCH-type models can be defined with \code{garchOrder},
which is a numeric vector with two elements. Its first element is the
ARCH order p, whereas the GARCH order q can be adjusted via the second
element. If no adjustments are made, the orders p = q = 1 are selected. The
number of out-sample observations is set via the argument \code{n.out}. If n
is the total number of observations of the
whole price series, the model is estimated for the first n - n.out
observations (in-sample), while the VaR and the ES are obtained for the last
n.out observations (out-sample) based on the estimated model for the
in-sample. Moreover, the data-driven estimation method of the underlying
scale function can be adjusted via the argument \code{smooth}. If
\code{smooth = 'lpr'} is selected, the scale function is obtained by
applying an iterative plug-in algorithm logarithm of the squared
centralized returns. Depending on the setting of \code{model} an algorithm
proposed by Feng, Gries and Fritz (2020) or by Letmathe, Feng and Uhde
(2021) is employed. In the former case, the function \code{msmooth()} of the
\code{smoots} package is applied and for the latter the \code{tsmoothlm()}
function of the \code{esemifar} package is used. An ellipsis \code{...} is
implemented to allow for additional arguments for \code{msmooth()} and
\code{tsmoothlm()}.

NOTE:

This function makes use of the \code{arima()} function of the stats package,
of the \code{fracdiff()} function of the \code{fracdiff} package, of the
\code{ugarchspec()} and \code{ugarchfit()} functions of the \code{rugarch}
package, of the \code{msmooth()} function of the \code{smoots} package
and of the \code{esemifar()} function of the \code{esemifar} for estimation.
Moreover, Log-GARCH and FI-Log-GARCH models in the parametric part of the
complete models are estimated via their ARMA and FARIMA representations,
respectively, and must therefore satisfy \eqn{p \geq q}.
}
\examples{

# Example for Walmart Inc. (WMT)
prices = WMT$price.close

# forecasting VaR and ES
results = varcast(prices, model = 'sGARCH', n.out = 250)
ret.out = results$ret.out
n.out = length(ret.out)
VaR97.5 = results$VaR.e
VaR99 = results$VaR.v
ES = results$ES

# plotting VaR at 99\% coverage
matplot(1:n.out, cbind(-ret.out, VaR99),
  type = 'hl',
  xlab = 'number of out-of-sample obs.', ylab = 'losses, VaR and ES',
  main = '99\% VaR (red) for the WMT return series')

# plotting VaR at 97.5\% coverage and corresponding ES
matplot(1:n.out, cbind(-ret.out, ES, VaR97.5),
  type = 'hll',
  xlab = 'number of out-of-sample obs.', ylab = 'losses, VaR and ES',
  main = '97.5\% VaR (green) and ES (red) for the WMT return series')

}
\references{
Baillie, R. T., Bollerslev, T., & Mikkelsen, H. O. (1996). Fractionally
integrated generalized autoregressive conditional heteroskedasticity.
In: Journal of Econometrics, 74.1, pp. 3-30.

Bollerslev, T. (1986) Generalized autoregressive conditional
heteroskedasticity. In: Journal of Econometrics 31.3, pp. 307-327.

Ding, Z., Granger, C.W., and Engle, R.F. (1993). A long memory property
of stock market returns and a new model. In: Journal of Empirical Finance
1.1, pp. 83-106.

Feng, Y. (2004). Simultaneously modeling conditional heteroskedasticity and
scale change. In: Econometric Theory 20.3, pp. 563-596.

Feng, Y., Beran, J., Letmathe, S., & Ghosh, S. (2020). Fractionally
integrated Log-GARCH with application to value at risk and expected
shortfall (No. 137). Paderborn University, CIE Center for
International Economics.

Pantula, S.G. (1986). Modeling the persistence of conditional variances:
a comment. In: Econometric Reviews 5, pp. 79-97.

Geweke, J. (1986). Comment on: Modelling the persistence of conditional
variances. In: Econometric Reviews 5, pp. 57-61.

Letmathe, S., Feng, Y., & Uhde, A. (2021). Semiparametric GARCH models with
long memory applied to Value at Risk and Expected Shortfall (No. 141).
Paderborn University, CIE Center for International Economics.

McNeil, A.J., Frey, R., and Embrechts, P. (2015). Quantitative risk
management: concepts, techniques and tools - revised edition. Princeton
University Press.

Milhoj, A. (1988). A Multiplicative parametrization of ARCH models.
Universitetets Statistiske Institut.

Nelson, D. B. (1991). Conditional heteroskedasticity in asset returns: A
new approach. In: Econometrica: Journal of the Econometric Society, 347-370.
}
\author{
\itemize{
\item Sebastian Letmathe (Scientific Employee) (Department of Economics,
Paderborn University) \cr
\item Dominik Schulz (Scientific Employee) (Department of Economics,
Paderborn University), \cr
}
}
