% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/makeDisplay.R
\name{makeDisplay}
\alias{makeDisplay}
\title{Create a Trelliscope Display}
\usage{
makeDisplay(data, name, group = "common", desc = "", mdDesc = NULL,
  height = 500, width = 500, panelFn = NULL, lims = list(x = "free", y =
  "free", prepanelFn = NULL), cogFn = NULL, state = NULL,
  preRender = FALSE, thumbIndex = 1, cogConn = dfCogConn(),
  output = NULL, conn = getOption("vdbConn"), verbose = TRUE,
  keySig = NULL, params = NULL, packages = NULL, control = NULL,
  detectGlobals = TRUE)
}
\arguments{
\item{data}{data of class "ddo" or "ddf" (see \code{\link{ddo}}, \code{\link{ddf}})}

\item{name}{the name of the display (no special characters, spaces are converted to underscores)}

\item{group}{the group the display belongs to, where displays are organized into groups (no special characters, spaces are
converted to underscores).  Defaults to "common"}

\item{desc}{a description of the display (used in the viewer)}

\item{mdDesc}{an optional longer-form description of the display and data, which can be text or can be a path to a markdown file or file with html snippets.  The description will appear in the "Display Information" panel in the Trelliscope viewer.}

\item{height}{reference dimensions (in pixels) for each panel (panels will be resized based on available space in the viewer)}

\item{width}{reference dimensions (in pixels) for each panel (panels will be resized based on available space in the viewer)}

\item{panelFn}{a function that produces a plot and takes one argument, which will be the current split of the data being passed to it.  It is recommended that you first test \code{panelFn} on a single key-value pair using \code{panelFn(data[[1]][[2]])}. This function must return either an object of class "ggplot", "trellis", or return "NULL" (for base plot commands)}

\item{lims}{either an object of class "trsLims" as obtained from \code{\link{setLims}} or a list with elements x, y, and prepanelFn, that specify how to apply \code{\link{prepanel}} and \code{\link{setLims}}}

\item{cogFn}{a function that returns a named list, where each element of the list is a cognostic feature (with length 1). This list must be coerceable to a 1-row data frame. The function should take one argument, which will be the current split of the data being passed to it.  Useful to test with \code{cogFn(divExample(dat))}}

\item{state}{if specified, this tells the viewer the default parameter settings (such as layout, sorting, filtering, etc.) to use when the display is viewed (see \code{\link{validateState}} for details)}

\item{preRender}{should the panels be pre-rendered and stored (\code{TRUE}), or rendered on-the-fly (\code{FALSE}, default)?  Default is recommended unless rendering is very expensive.  See Details.}

\item{thumbIndex}{the index value to use for creating the thumbnail}

\item{cogConn}{a connection to store the cognostics data.  By default, this is \code{\link{dfCogConn}()}.}

\item{output}{how to store the panels and metadata for the display (unnecessary to specify in most cases -- see details)}

\item{conn}{VDB connection info, typically stored in options("vdbConn") at the beginning of a session, and not necessary to specify here if a valid "vdbConn" object exists}

\item{verbose}{print status messages?}

\item{keySig}{a user-defined key signature (string - see details)}

\item{params}{a named list of objects external to the input data that are needed in the distributed computing (most should be taken care of automatically such that this is rarely necessary to specify)}

\item{packages}{a vector of R package names that contain functions used in \code{panelFn} or \code{cogFn} (most should be taken care of automatically such that this is rarely necessary to specify)}

\item{control}{parameters specifying how the backend should handle things (most-likely parameters to \code{rhwatch} in RHIPE) - see \code{\link[datadr]{rhipeControl}} and \code{\link[datadr]{localDiskControl}}}

\item{detectGlobals}{if TRUE params are automatically detected (packages are always auto-detected)}
}
\description{
Create a trelliscope display and add it to a visualization database (VDB)
}
\details{
Many of the parameters are optional or have defaults.  For several examples, see the documentation at deltarho.org: \url{http://deltarho.org/docs-trelliscope}

Panels by default are not pre-rendered. Instead, this function creates a display object and computes and stores the cognostics.  Panels are then rendered on the fly by the DeltaRho backend and pushed to the Trelliscope viewer as html with the panel images embedded in the html.  If a user would like to pre-render the images for every subset (using \code{preRender = TRUE}), then by default the image files for the panels will be stored to a local disk connection (see \code{\link{localDiskConn}}) inside the VDB directory, organized in subdirectories by group and name of the display.  Optionally, the user can specify the \code{output} parameter to be any valid "kvConnection" object, as long as it is one that persists on disk (e.g. \code{\link{hdfsConn}}).

\code{keySig} does not generally need to be specified.  It is useful to specify when creating multiple displays that you would like to be treated as related displays, so that you can view them side by side.  Two displays are determined to be related when their key signatures, typically computed as a md5 hash of the complete collection of keys, match.  Sometimes two displays will have data where the keys match for a significant portion of subsets, but not all.  Manually specifying the same \code{keySig} for each can ensure that they will be treated as related displays.
}
\examples{
\dontrun{
library(ggplot2)

vdbConn(tempfile(), autoYes = TRUE)

# divide housing data by county
byCounty <- divide(housingData::housing, by = c("county", "state"))

xlim <- as.Date(c("2008-01-31", "2016-01-31"))

# plot list price vs. time for each county
makeDisplay(byCounty, name = "county_time",
  panelFn = function(x)
    ggplot(x, aes(time, medListPriceSqft)) +
      geom_point() + xlim(xlim))

# divide housing data by state
byState <- divide(housingData::housing, by = "state")

# create a "displayHref" cognostic that links to the by county display
# filtered down to all counties in the current state
cogFn <- function(x) {
  state <- stateSpec(
    name = "county_time",
    sort = list(county = "asc"),
    layout = list(nrow = 2, ncol = 4),
    filter = list(state = list(select = getSplitVar(x, "state"))))

  list(countyPlots = cogDisplayHref(state = state, defLabel = TRUE))
}

# plot distribution of list price vs. time for each state
makeDisplay(byState, name = "state_time_CI",
  panelFn = function(x)
    ggplot(x, aes(time, medListPriceSqft)) +
      stat_summary(fun.data = "mean_cl_boot") + xlim(xlim),
  cogFn = cogFn)

# open up the state display
# try clicking on the link for "countyPlots"
# the by county display will be loaded filtered to the state
view("state_time_CI")
}

}
\seealso{
\code{\link{prepanel}}, \code{\link{setLims}}, \code{\link{divide}}
}

