% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/semidiscrete1.R
\name{semidiscrete1}
\alias{semidiscrete1}
\title{Compute Semidiscrete Optimal Transport for Euclidean Distance Cost}
\usage{
semidiscrete1(source, target, xrange = c(0, 1), yrange = c(0, 1),
  verbose = FALSE, reg = 0)
}
\arguments{
\item{source}{A matrix specifing the source measure.}

\item{target}{A three-column matrix specifing the target measure in the form
x-coordinate, y-coordinate, mass.}

\item{xrange, yrange}{Vectors with two components defining the window on which 
the source measure lives. Defaults to \eqn{[0,1] \times [0,1]}{[0,1] x [0,1]}.
\code{source} is interpreted as an image of equally sized quadratic pixels
on this window.}

\item{verbose}{Logical. Shall information about multiscale progress and L-BFGS return
codes be printed?}

\item{reg}{A non-negative regularization parameter. It is usually not
necessary to deviate from the default 0.}
}
\value{
A list describing the solution. The components are
  \item{weights}{A vector of length equal to the first dimension of \code{target}
        containing the weights for the Apollonius diagram discribing the
        optimal semidiscrete transport from source to target.}
  \item{wasserstein_dist}{The \eqn{L_1}{L1}-Wasserstein distance between source and target.}
  \item{ret_code}{A return code. Equal to 1 if everything is OK, since our code
        interrupts the usual lbfgs code. Other values can be converted to the
        corresponding return message by using \code{\link{ret_message}}.}
}
\description{
Computes the weight vector of the Apollonius diagram describing the semidiscrete
optimal transport plan for the Euclidean distance cost function and the associated
Wasserstein distance.
}
\note{
This function requires the Computational Geometry Algorithms Library (CGAL),
      available at \url{https://www.cgal.org}. Adapt the file src/Makevars according
      to the instructions given there and re-install from source.
      
      Internally the code from liblbfgs 1.10 by Naoaki Okazaki (2010) is used.
      See \url{http://www.chokkan.org/software/liblbfgs/}.
      
      A stand-alone version of the C++ code of this function is available
      at \url{https://github.com/valentin-hartmann-research/semi-discrete-transport}.
}
\examples{
\dontrun{
# the following function rotates a matrix m clockwise, so
# that image(rococlock(m)) has the same orientation as print(m):
roclock <- function(m) t(m)[, nrow(m):1]

set.seed(30)
n <- 20
nu <- matrix(c(runif(2*n), rgamma(n,3,1)), n, 3)
pixelbdry <- seq(0,1,length=33)
image(pixelbdry, pixelbdry, roclock(random32a$mass), asp=1, col = grey(seq(0,1,length.out=32)))
points(nu[,1], nu[,2], pch=16, cex=sqrt(nu[,3])/2, col=2)

res <- semidiscrete1(random32a$mass, nu)
plot_apollonius(nu[,1:2], res$weights, show_weights = FALSE, add = TRUE)
points(nu[,1], nu[,2], pch=16, cex=sqrt(nu[,3])/2, col=2)}

}
\references{
V. Hartmann and D. Schuhmacher (2017).
            Semi-discrete optimal transport --- the case p=1.
            Preprint \href{https://arxiv.org/abs/1706.07650}{arXiv:1706.07650}
            
            Menelaos Karavelas and Mariette Yvinec. 2D Apollonius Graphs 
            (Delaunay Graphs of Disks). In CGAL User and Reference Manual.
            CGAL Editorial Board, 4.12 edition, 2018
            
            Naoaki Okazaki (2010). libLBFGS: a library of Limited-memory
            Broyden-Fletcher-Goldfarb-Shanno (L-BFGS). Version 1.10
}
\seealso{
\code{\link{ret_message}}, \code{\link{semidiscrete}}.
}
\author{
Valentin Hartmann \email{valentin.hartmann@epfl.ch} (stand-alone C++ code)\cr
        Dominic Schuhmacher \email{schuhmacher@math.uni-goettingen.de} (R-port)
}
\keyword{optimal}
\keyword{semidiscrete,}
\keyword{transport}
