% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/computeGraph.R
\name{computeGraphMetric}
\alias{computeGraphMetric}
\title{Compute top vertices by the metric values on a graph.}
\usage{
computeGraphMetric(channel, graph, type = "degree", top = 10,
  rankFunction = "rank", weight = FALSE, vertexWhere = graph$vertexWhere,
  edgeWhere = graph$edgeWhere, keyAsFactor = TRUE, allTables = NULL,
  test = FALSE, ...)
}
\arguments{
\item{channel}{connection object as returned by \code{\link{odbcConnect}}}

\item{graph}{an object of class \code{'toagraph'} referencing graph 
tables in Aster database.}

\item{type}{choose between graph metrics to compute: \code{'degree', 'in-degree', 
'out-degree', 'clustering', 'shortestpath', 'pagerank', 'betweenness', 
'eigenvector', 'closeness', 'avg-closeness', 'k-degree', 'alt-closeness'}.}

\item{top}{the number of vertices to return. If \code{top >= 0} then \code{top} vertices 
sorted by the metric value are returned, otherwise all vertices are returned. 
Returned vertices are ordered by the computed graph metric only when \code{top >= 0}.}

\item{rankFunction}{one of \code{rownumber, rank, denserank, percentrank}. Rank computed and
returned for each vertex and each metric type. \code{rankFunction} determines which SQL window 
function computes vertex rank value (default \code{rank} corresponds to SQL \code{RANK()} window function). 
When threshold \code{top} is greater than 0 ranking function used to limit number of 
vertices returned (see details).}

\item{weight}{logical or character: if logical then \code{TRUE} indicates using \code{'weight'} edge
attribute, otherwise no weight used. If character then use as a name for the edge weight attribute. 
The edge weight may apply with types \code{'clustering', 'shortestpath'} and centrality measures.}

\item{vertexWhere}{SQL WHERE clause limiting data from the vertex table. This value when not null
overrides corresponding value \code{vertexWhere} from \code{graph} (use SQL as if in WHERE clause but 
omit keyword WHERE).}

\item{edgeWhere}{SQL WHERE clause limiting data from the edge table. This value when not null
overrides corresponding value \code{edgeWhere} from \code{graph} (use SQL as if in WHERE clause but 
omit keyword WHERE).}

\item{keyAsFactor}{logical: should key column be converted to factor? If \code{TRUE} then conversion
will always take place for any of integer, numeric, or character data types.}

\item{allTables}{pre-built information about existing tables.}

\item{test}{logical: if TRUE show what would be done, only (similar to parameter \code{test} in \pkg{RODBC} 
functions: \link{sqlQuery} and \link{sqlSave}).}

\item{...}{other arguments passed on to Aster graph functions except for \code{EDGEWEIGHT} argument -
use argument \code{weight} instead. Aster function areguments are not case-sensetive.}
}
\value{
dataframe containing one vertice per row with key value, computed metric value, and its rank 
  using \code{rankFunction}.
}
\description{
Compute top vertices by the metric values on a graph.
}
\examples{
if(interactive()) {
library(ggplot2)

policeGraphUn = toaGraph("dallaspolice_officer_vertices", "dallaspolice_officer_edges_un", 
                         directed = FALSE, key = "officer", 
                         source = "officer1", target = "officer2", 
                         vertexAttrnames = c("offense_count"), edgeAttrnames = c("weight"))
               
# initialize connection to Lahman baseball database in Aster 
conn = odbcDriverConnect(connection="driver={Aster ODBC Driver};
                         server=<dbhost>;port=2406;database=<dbname>;uid=<user>;pwd=<pw>")
                         
createTopMetricPlot <- function(data, metric, xlab='Officer', ylab='Degree', title) {
   p = ggplot(data) +
       geom_bar(aes_string("key", metric, fill="key"), stat='identity') +
       labs(x=xlab,y=ylab,title=title) +
       ggthemes::theme_tufte() + 
       theme(legend.position='none', 
             axis.text.x = element_text(size=16, angle = 315, vjust = 1), 
             plot.title = element_text(size=20),
             axis.ticks = element_blank())
   
   return(p)
}

# top degree officers
topDegree = computeGraphMetric(conn, policeGraphUn, type="degree", top=30)
createTopMetricPlot(topDegree, 'degree', ylab='Degree', title='Top 30 Officers by Degree') 

# top betweenness officers
topbetweenness = computeGraphMetric(conn, policeGraphUn, type='betweenness', top=25)
createTopMetricPlot(topbetweenness, 'betweenness', ylab='Betweenness', 
                    title='Top 25 Officers (Betweenness)')                     
                         
}
}

