# Dichtefunktion und Verteilung einer multivariate truncated normal
#
# Problem ist die Bestimmung der Randverteilung einer Variablen.
#
# 1. Im bivariaten Fall kann explizit eine Formel angegeben werden (vgl. Arnold (1993))
# 2. Im multivariaten Fall kann ein Integral angegeben werden (vgl. Horrace (2005))
# 3. Bestimmung der Dichtefunktion ber das Integral mglich?
# 4. Kann die Verteilungsfunktion pmvnorm() helfen? Kann man dann nach einer Variablen differenzieren?

# Literatur:
#
# Genz, A. (1992). Numerical computation of multivariate normal probabilities. Journal of Computational and Graphical Statistics, 1, 141150
# Genz, A. (1993). Comparison of methods for the computation of multivariate normal probabilities. Computing Science and Statistics, 25, 400405
# Horrace (2005).
# Jack Cartinhour (1990): One-dimensional marginal density functions of a truncated multivariate normal density function
# Communications in Statistics - Theory and Methods, Volume 19, Issue 1 1990 , pages 197 - 203

library(mvtnorm)

# Dichtefunktion fr Randdichte f(xn) einer Truncated Multivariate Normal Distribution,
# vgl. Jack Cartinhour (1990) "One-dimensional marginal density functions of a truncated multivariate normal density function"
#
# @param xn Vektor der Lnge l von Punkten, an dem die Randdichte ausgewertet wird
# @param i Index  (1..n) dessen Randdichte berechnet werden soll
# @param mean (nx1) Mittelwertvektor
# @param sigma (nxn)-Kovarianzmatrix
# @param lower,upper Trunkierungsvektor lower <= x <= upper
dtmvnorm.marginal <- function(xn, n=1, mean=rep(0, nrow(sigma)), sigma=diag(length(mean)), lower=rep(-Inf, length = length(mean)), upper=rep( Inf, length = length(mean)))
{
   #if (!is.scalar(xn))
   #{
   #  stop("xn must be a scalar")
   #}
   
   if (NROW(sigma) != NCOL(sigma)) {
     stop("sigma must be a square matrix")
   }

   if (length(mean) != NROW(sigma)) {
    stop("mean and sigma have non-conforming size")
   }
   
   # Anzahl der Dimensionen                
   k = length(mean)
   
   if (n < 1 || n > length(mean) || !is.numeric(n) || length(n) > 1 ||  !n %in% 1:length(mean))
   {
     stop("n must be a integer scalar in 1..length(mean)")
   }
   
   # TODO : funktioniert nicht hier wg. Vektorisierung: a[n] <= x <= b[n], sonst Dichte 0
   #if (!(lower[n]<=xn && xn<=upper[n]))
   #{
   #  return(0)
   #}
   
   # Univariater Fall, vgl. Greene (2003), S.573
   if (k == 1)
   {
     prob <- pnorm(upper, mean=mean, sd=sqrt(sigma)) - pnorm(lower, mean=mean, sd=sqrt(sigma))
     return(
     ifelse(
       lower[1]<=xn & xn<=upper[1], 
       dnorm(xn, mean=mean, sd=sqrt(sigma)) / prob,
       0
     ))
   }

   # Kovarianzmatrix
   C = sigma
    
   # Inverse Kovarianzmatrix
   A = solve(sigma)

   # Partitionierung von A und C
   A_1  = A[-n,-n] # (n-1) x (n-1)
   #a_nn = A[n, n]  # 1x1
   #a    = A[-n, n] # (n-1) x 1
   A_1_inv = solve(A_1)
   
   C_1  = C[-n,-n] # (n-1) x (n-1)
   c_nn = C[n, n]  #  1x1
   c    = C[-n, n] # (n-1) x 1
   
   # Partitionierung von Mittelwertvektor mu
   mu   = mean
   mu_1 = mean[-n]
   mu_n = mean[n]
   
   # Skalierungsfaktor der Dichte
   p = pmvnorm(lower=lower, upper=upper, mean=mu, sigma=C)

   f_xn <- c()
   for (i in 1:length(xn))
   {
     if (!(lower[n]<=xn[i] && xn[i]<=upper[n]) || is.infinite(xn[i]))
     {
       f_xn[i] = 0
       next
     }
     
     # m(x_n) --> (n-1x1)
     # Aufpassen bei z.B. m=c(Inf, Inf, NaN) und c=0
     m = mu_1 + (xn[i] - mu_n) * c / c_nn
     
     f_xn[i] =  exp(-0.5*(xn[i]-mu_n)^2/c_nn) * pmvnorm(lower=lower[-n], upper=upper[-n], mean=m, sigma=A_1_inv)
   }
   1/p * 1/sqrt(2*pi*c_nn) * f_xn
}


# Bestimme die k1-dimensionale Randverteilungen durch numerische Integration mit adapt() und integrate()
#
# @param x Vektor der Lnge k1 < k fr die Randverteilung
# @param n Vektor der Lnge k1 < k fr die Randverteilung mit den Indizes (1..k)
dtmvnorm.marginal.integration <- function(x, n, mean, sigma, lower, upper)
{
  k  = length(mean)
  
  # Anzahl der Dimensionen der Randdichte
  k1 = length(x)
  # Anzahl der Dimensionen/Variablen, in denen integriert werden soll/die rausintegriert werden sollen k1 + k2 = k
  k2 = k - k1
  
  if (k1 >= k || k1 < 1) stop("length of x must be 1 <= x < k")
  if (length(x) != length(n)) stop("length of x must be the same as length of n")
  
  # Skalierungsfaktor der Dichte
  p = pmvnorm(lower=lower, upper=upper, mean=mu, sigma=sigma)
  
  # Dichtefunktion in k = k1 + k2 Dimensionen
  fd = function(z)
  {
    y     = numeric(k)
    y[n]  = x   # k1 Dimensionen der Randdichte
    y[-n] = z   # k2 Dimensionen: Variablen, die rausintegriert werden
    
    res = dtmvnorm(y, mean, sigma, lower, upper)                  
    cat("Evaluating function for z=",z," res=",res,"\n")
    flush.console()
    return(res)
  }
  
  fdv = Vectorize(fd)
  
  # Integriere in k2 = k - k1 dimensionen, wenn k2 = 1, dann mit integrate(), sonst mit adapt()
  if (k2 == 1)
  {
    res = integrate(fdv, lower=lower[-n], upper=upper[-n])
  }
  else
  {
    res = adapt(ndim=k2, lo = lower[-n], up = upper[-n], functn = fd)
  }
  return(res/p) 
}


#dtmvnorm.marginal.integration(x=c(0,0), n=c(1,2), mean, sigma, lower=a, upper=b)
#dtmvnorm.marginal.integration(x=0, n=1, mean, sigma, lower=a, upper=b)
#dtmvnorm.marginal(x=0, n=1, mean, sigma, lower=a, upper=b)
#
#dtmvnorm.marginal.integration(x=c(0,0), n=c(1,2), mean, sigma, lower=a, upper=b)


