#' Spanning labels to identify groups of rows or columns
#'
#' @export
#' @inheritParams tt
#' @inheritParams style_tt
#' @param i A vector of labels with length equal to the number of rows in `x`, or a named list of row indices to group. The names of the list will be used as labels. The indices represent the position where labels should be inserted in the original table. For example,
#' + `i=list("Hello"=5)`: insert the "Hello" label after the 4th row in the original table.
#' + `i=list("Hello"=2, "World"=2)`: insert the two labels consecutively after the 1st row in the original table.
#' + `i=list("Foo Bar"=0)`: insert the label in the first row after the header.
#' @param j String or named list
#' - Named list of column indices to group, ex: `j=list("A"=1:2,"B"=3:6)`. The names of the list will be used as labels. See below for more examples. Note: empty labels must be a space: " ".
#' - A single string when column names include the group name as a prefix, ex: group1_column1, group1_column2, etc.
#' @param ... Other arguments are ignored.
#' @return An object of class `tt` representing the table.
#' @param indent integer number of `pt` to use when indenting the non-labelled rows.
#' @template limitations_word_markdown
#' @details
#' Warning: The `style_tt()` can normally be used to style the group headers, as expected, but that feature is not available for Markdown and Word tables.
#' @examples
#'
#' # vector of row labels
#' dat <- data.frame(
#'   label = c("a", "a", "a", "b", "b", "c", "a", "a"),
#'   x1 = rnorm(8),
#'   x2 = rnorm(8)
#' )
#' tt(dat[, 2:3]) |> group_tt(i = dat$label)
#'
#' # named lists of labels
#' tt(mtcars[1:10, 1:5]) |>
#'   group_tt(
#'     i = list(
#'       "Hello" = 3,
#'       "World" = 8
#'     ),
#'     j = list(
#'       "Foo" = 2:3,
#'       "Bar" = 4:5
#'     )
#'   )
#'
#' dat <- mtcars[1:9, 1:8]
#' tt(dat) |>
#'   group_tt(i = list(
#'     "I like (fake) hamburgers" = 3,
#'     "She prefers halloumi" = 4,
#'     "They love tofu" = 7
#'   ))
#'
#' tt(dat) |>
#'   group_tt(
#'     j = list(
#'       "Hamburgers" = 1:3,
#'       "Halloumi" = 4:5,
#'       "Tofu" = 7
#'     )
#'   )
#'
#' x <- mtcars[1:5, 1:6]
#' tt(x) |>
#'   group_tt(j = list("Hello" = 1:2, "World" = 3:4, "Hello" = 5:6)) |>
#'   group_tt(j = list("Foo" = 1:3, "Bar" = 4:6))
#'
#' # column names with delimiters
#' dat <- data.frame(
#'   A_id = 1,
#'   A_a1 = 2,
#'   A_a2 = "3",
#'   B_b1 = 4,
#'   B_b2 = 5,
#'   B_C = 6
#' )
#' tt(dat) |> group_tt(j = "_")
#'
group_tt <- function(
    x,
    i = getOption("tinytable_group_i", default = NULL),
    j = getOption("tinytable_group_j", default = NULL),
    indent = getOption("tinytable_group_indent", default = 1),
    ...) {
  # ... is important for ihead passing

  if (!inherits(x, "tinytable")) {
    stop("`x` must be generated by `tinytable::tt()`.", call. = FALSE)
  }
  if (is.null(i) && is.null(j)) {
    stop("At least one of `i` or `j` must be specified.", call. = FALSE)
  }
  assert_integerish(indent, lower = 0)

  if (isTRUE(check_string(j))) {
    if (any(grepl(j, x@names, fixed = TRUE))) {
      j_delim <- j_delim_to_named_list(x = x, j = j)
      x@names <- j_delim$colnames
      j <- j_delim$groupnames
    } else {
      j <- NULL
    }
  }

  if (is.null(i) && is.null(j)) {
    return(x)
  }

  # vector of labels
  if (isTRUE(check_atomic_vector(i))) {
    i <- sanitize_group_vec2list(i)
  }

  i <- sanitize_group_index(i, hi = nrow(x) + 1, orientation = "row")
  j <- sanitize_group_index(j, hi = ncol(x), orientation = "column")

  # increment indices eagerly
  i <- unlist(i)

  if (!is.null(i)) {
    if (x@group_n_i > 0) {
      stop(
        "Only one row-wise `group_tt(i = ...)` call is allowed.",
        call. = FALSE
      )
    }

    x@group_n_i <- length(i)
    x@nrow <- x@nrow + x@group_n_i
    x@group_index_i <- as.numeric(i) + cumsum(rep(1, length(as.numeric(i)))) - 1

    if (isTRUE(indent > 0)) {
      idx_indent <- setdiff(seq_len(nrow(x)), i + seq_along(i) - 1)
      x <- style_tt(x, i = idx_indent, j = 1, indent = indent)
    }
  }

  if (!is.null(j)) {
    x@nhead <- x@nhead + 1
  }

  # apply group labels lazily
  cal <- call("group_eval", i = i, j = j, indent = indent)
  x@lazy_group <- c(x@lazy_group, list(cal))

  return(x)
}

sanitize_group_vec2list <- function(vec) {
  if (is.factor(vec)) vec <- as.character(vec)
  rle_result <- rle(vec)
  idx <- cumsum(c(1, utils::head(rle_result$lengths, -1)))
  idx <- as.list(idx)
  names(idx) <- rle_result$values
  return(idx)
}

sanitize_group_index <- function(idx, hi, orientation) {
  if (is.null(idx)) {
    return(idx)
  }
  assert_list(idx, named = TRUE)
  for (n in names(idx)) {
    if (orientation == "row") {
      assert_integerish(idx[[n]], len = 1, lower = 1, upper = hi, name = n)
    } else {
      assert_integerish(idx[[n]], lower = 1, upper = hi, name = n)
    }
  }
  # allow duplicated indices for consecutive rows
  # if (anyDuplicated(unlist(idx)) > 0) stop("Duplicate group indices.", call. = FALSE)
  out <- lapply(idx, function(x) min(x):max(x))
  return(out)
}


j_delim_to_named_list <- function(x, j) {
  nm <- x@names

  # Find which elements contain the delimiter, and optionally j. Others are left as is
  check_for_multiple_delims <- any(
    lengths(gregexec(pattern = j, text = nm)) > 1L
  )
  if (check_for_multiple_delims) {
    warning(
      "Multiple delimiters found in column names. Only the first delimiter will be used for grouping."
    )
  }

  indices <- grepl(j, nm, fixed = TRUE)
  groupnames <- sub(
    pattern = paste0(j, ".*"),
    replacement = "",
    x = nm[indices]
  )
  indices <- which(grepl(j, nm, fixed = TRUE))
  groupnames <- split(indices, groupnames)

  # Extract suffixes (after delimiter) used as new sub-headers
  colnames <- sub(pattern = paste0(".*?", j), replacement = "", x = nm)

  out <- list(colnames = colnames, groupnames = groupnames)
  return(out)
}
