% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AllGenerics.R, R/refine.R, R/seriate.R
\docType{methods}
\name{seriation}
\alias{seriation}
\alias{seriate}
\alias{seriate-method}
\alias{permute}
\alias{permute-method}
\alias{refine}
\alias{refine-method}
\alias{refine,CountMatrix-method}
\alias{seriate,CountMatrix,missing-method}
\alias{seriate,IncidenceMatrix,missing-method}
\alias{seriate,CountMatrix,BootCA-method}
\alias{permute,CountMatrix,PermutationOrder-method}
\alias{permute,IncidenceMatrix,PermutationOrder-method}
\title{Matrix seriation}
\usage{
seriate(object, subset, ...)

permute(object, order, ...)

refine(object, ...)

\S4method{refine}{CountMatrix}(object, cutoff, n = 1000, axes = c(1,
  2), ...)

\S4method{seriate}{CountMatrix,missing}(object,
  method = c("correspondance", "reciprocal"), EPPM = FALSE,
  margin = c(1, 2), stop = 100, ...)

\S4method{seriate}{IncidenceMatrix,missing}(object,
  method = c("correspondance", "reciprocal"), margin = c(1, 2),
  stop = 100, ...)

\S4method{seriate}{CountMatrix,BootCA}(object, subset, margin = c(1, 2),
  ...)

\S4method{permute}{CountMatrix,PermutationOrder}(object, order)

\S4method{permute}{IncidenceMatrix,PermutationOrder}(object, order)
}
\arguments{
\item{object}{An \eqn{m \times p}{m x p} data matrix.}

\item{subset}{A \linkS4class{BootCA} object giving the subset of
\code{object} to be used.}

\item{...}{Further arguments passed to other methods.}

\item{order}{A \linkS4class{PermutationOrder} object giving the permutation
order for rows and columns.}

\item{cutoff}{A function that takes a numeric vector as argument and returns
a single numeric value (see details).}

\item{n}{A non-negative \code{\link{integer}} giving the number of partial
bootstrap replications (see details).}

\item{axes}{A \code{\link{numeric}} vector giving the subscripts of the CA
axes to use (see details).}

\item{method}{A \code{\link{character}} string specifiying the method to be
used. This must be one of "\code{reciprocal}", "\code{correspondance}"
(see details). Any unambiguous substring can be given.}

\item{EPPM}{A \code{\link{logical}} scalar: should the seriation be computed
on EPPM instead of raw data?}

\item{margin}{A \code{\link{numeric}} vector giving the subscripts which the
rearrangement will be applied over: \code{1} indicates rows, \code{2}
indicates columns, \code{c(1, 2)} indicates rows then columns,
\code{c(2, 1)} indicates columns then rows.}

\item{stop}{A length-one \code{\link{numeric}} vector giving the stopping rule
(i.e. maximum number of iterations) to avoid infinite loop.}
}
\value{
\code{seriate} returns a \linkS4class{PermutationOrder} object.

 \code{permute} returns either a
 \linkS4class{CountMatrix}, \linkS4class{FrequencyMatrix} or
 \linkS4class{IncidenceMatrix} (the same as \code{object}).

 \code{refine} returns a \linkS4class{BootCA} object containing the subscript
 of samples to be kept (i.e. samples with maximum dimension length of the
 convex hull smaller than the cutoff value).
}
\description{
\code{seriate} computes a permutation order for rows and/or columns.

 \code{permute} rearranges a data matrix according to a permutation order.

 \code{refine} performs a partial bootstrap correspondance analysis
 seriation refinement.
}
\section{Seriation}{

 The matrix seriation problem in archaeology is based on three conditions
 and two assumptions, which Dunell (1970) summarizes as follows.

 The homogeneity conditions state that all the groups included in a
 seriation must:
 \enumerate{
  \item{Be of comparable duration.}
  \item{Belong to the same cultural tradition.}
  \item{Come from the same local area.}
 }

 The mathematical assumptions state that the distribution of any historical
 or temporal class:
 \enumerate{
  \item{Is continuous through time.}
  \item{Exhibits the form of a unimodal curve.}
 }
 Theses assumptions create a distributional model and ordering is
 accomplished by arranging the matrix so that the class distributions
 approximate the required pattern. The resulting order is infered
 to be chronological.

 The following seriation methods are available:
 \describe{
  \item{correspondance}{Correspondance analysis-based seriation.
  Correspondance analysis (CA) is an effective method for the seriation of
  archaeological assemblages. The order of the rows and columns is given by
  the coordinates along one dimension of the CA space, assumed to account
  for temporal variation. The direction of temporal change within the
  correspondance analysis space is arbitrary: additional information is
  needed to determine the actual order in time.}
  \item{reciprocal}{Reciprocal ranking (incidence data) or averaging
  (frequency data) seriation. These procedures iteratively rearrange rows
  and/or columns according to their weighted rank in the data matrix until
  convergence. Note that this procedure could enter into an infinite loop.
  If no convergence is reached before the maximum number of iterations, it
  stops with a warning.}
 }
}

\section{CA seriation refining}{

 \code{refine} allows to identify samples that are subject to sampling error
 or samples that have underlying structural relationships and might be
 influencing the ordering along the CA space.

 This relies on a partial bootstrap approach to CA-based seriation where each
 sample is replicated \code{n} times. The maximum dimension length of
 the convex hull around the sample point cloud allows to remove samples for
 a given \code{cutoff} value.

 According to Peebles and Schachner (2012), "[this] point removal procedure
 [results in] a reduced dataset where the position of individuals within the
 CA are highly stable and which produces an ordering consistend with the
 assumptions of frequency seriation."

 If the results of \code{\link{refine}} is used as an input argument in
 \code{seriate}, a correspondance analysis is performed on the subset of
 \code{object} which matches the samples to be kept. Then excluded samples
 are projected onto the dimensions of the CA coordinate space using the row
 transition formulae. Finally, row coordinates onto the first dimension
 give the seriation order.

 To monitor the execution of these re-sampling procedure, a progress bar
 will automatically be displayed if the \code{\link[pbapply]{pbapply}}
 package is installed on your machine.
}

\examples{
# Matrix seriation
# Reproduces Desachy 2004 results
## Coerce dataset to abundance matrix
compiegne_count <- as(compiegne, "CountMatrix")

## Get seriation order for columns on EPPM using the reciprocal averaging method
## Expected column order: N, A, C, K, P, L, B, E, I, M, D, G, O, J, F, H
(compiegne_indices <- seriate(compiegne_count, method = "reciprocal",
                              EPPM = TRUE, margin = 2))

## Permute columns
compiegne_new <- permute(compiegne_count, compiegne_indices)

## Plot new matrix
plotBar(compiegne_new, EPPM = FALSE)

\donttest{
# Refine matrix seriation (this is a long running example)
# Reproduces Peeples and Schachner 2012 results
zuni_count <- as(zuni, "CountMatrix")

## Samples with convex hull maximum dimension length greater than the cutoff
## value will be marked for removal.
## Define cutoff as one standard deviation above the mean
fun <- function(x) { mean(x) + sd(x) }

## Get indices of samples to be kept
## Warning: this may take a few seconds!
set.seed(123)
(zuni_refined <- refine(zuni_count, cutoff = fun))

## Get CA-based seriation order
(zuni_indices <- seriate(zuni_count, zuni_refined, margin = 1))
}
}
\references{
Desachy, B. (2004). Le sériographe EPPM: un outil informatisé de sériation
 graphique pour tableaux de comptages. \emph{Revue archéologique de
 Picardie}, 3(1), 39-56.
 DOI: \href{https://doi.org/10.3406/pica.2004.2396}{10.3406/pica.2004.2396}.

 Dunnell, R. C. (1970). Seriation Method and Its Evaluation. \emph{American
 Antiquity}, 35(03), 305-319.
 DOI: \href{https://doi.org/10.2307/278341}{10.2307/278341}.

 Ihm, P. (2005). A Contribution to the History of Seriation in Archaeology.
 In C. Weihs & W. Gaul (Eds.), \emph{Classification: The Ubiquitous Challenge}
 (p. 307-316). Berlin Heidelberg: Springer.
 DOI: \href{https://doi.org/10.1007/3-540-28084-7_34}{10.1007/3-540-28084-7_34}.

 Peeples, M. A., & Schachner, G. (2012). Refining correspondence
 analysis-based ceramic seriation of regional data sets. \emph{Journal of
 Archaeological Science}, 39(8), 2818-2827.
 DOI: \href{https://doi.org/10.1016/j.jas.2012.04.040}{10.1016/j.jas.2012.04.040}.
}
\seealso{
\link{refine} \link[FactoMineR]{CA}
}
\author{
N. Frerebeau
}
