\name{check}
\alias{checkEquals}
\alias{checkEqualsNumeric}
\alias{checkIdentical}
\alias{checkTrue}
\alias{checkException}
\alias{DEACTIVATED}

\title{ SciViews R Unit check functions }
\description{
  These functions define the code of test functions. They are designed to check
  the result of some test calculation.
}

\usage{
checkEquals(target, current, msg = "", tolerance = .Machine$double.eps^0.5,
    checkNames = TRUE, \dots)
checkEqualsNumeric(target, current, msg = "",
	tolerance = .Machine$double.eps^0.5, \dots)
checkIdentical(target, current, msg = "")
checkTrue(expr, msg = "")
checkException(expr, msg = "", silent = getOption("svUnit.silentException"))
DEACTIVATED(msg = "")
}

\arguments{
  \item{current}{ An object created for comparison (not an S4 class object) }
  \item{target}{ A target object as reference fo comparison }
  \item{msg}{ An optional (short!) message to document a test. This message is
    stored in the log and printed in front of each test report }
  \item{tolerance}{	numeric >= 0. A numeric check does not fail if differences
    are smaller than `tolerance' }
  \item{checkNames}{ Flag, if \code{FALSE} the names attributes are set to
    \code{NULL} for both \code{current} and \code{target} before performing
	the check }
  \item{expr}{ Syntactically valid R expression which can be evaluated and must
    return a logical vector (\code{TRUE}|\code{FALSE}). A named expression is
	also allowed but the name is disregarded. In \code{checkException()}, expr
	is supposed to generate an error to pass the test }
  \item{silent}{ Flag passed on to try, which determines if the error message
    generated by the checked function is displayed at the R console. By default,
	it is \code{FALSE} }
  \item{\dots}{ Optional arguments passed to \code{all.equal()} or
    \code{all.equal.numeric()} }
}

\value{
  \code{TRUE} if the test succeeds, \code{FALSE} if it fails, possibly with a
  'result' attribute containing more information about the problem. This is
  very different from corresponding functions in 'RUnit' that stop with an
  error in case of test failure. Consequently, current functions do not require
  the complex evaluation framework designed in 'RUnit' for that reason.
}

\details{
  These check functions are equivalent to various methods of the class
  junit.framework.Assert of Java junit framework. They should be code-compatible
  with functions of same name in 'RUnit' 0.4.17, except for \code{checkTrue()}
  that is vectorized here, but accept only a scalar result in 'RUnit'. For
  scalar test, the behaviour of the function is the same in both packages.

  See \code{svTest()} for examples of utilisation of these functions in actual
  test cases attached to R objects.

  See also the note about S4 objects in the \code{checkTrue()} online help of
  the 'RUnit' package.
}

\author{ Philippe Grosjean <phgrosjean@sciviews.org> has adapted existing code
  in 'RUnit' by Thomas Konig, Klaus Junemann & Matthias Burger and ported it
  to 'svUnit' }

\seealso{ \code{\link{svTest}}, \code{\link{Log}},
  \code{\link{guiTestReport}}, \code{\link[RUnit]{checkTrue}} }

\examples{
library(datasets)	# Make sure 'datasets' package is loaded
clearLog()			# Clear the svUnit log

# All these tests are correct
(checkEquals(c("A", "B", "C"), LETTERS[1:3]))
(checkEqualsNumeric(1:10, seq(1, 10)))
(checkIdentical(iris[1:50, ], iris[iris$Species == "setosa",]))
(checkTrue(1 < 2))
(checkException(log("a")))
Log()	# See what's recorded in the log

# ... but these ones fail
(checkEquals("A", LETTERS[1:3]))
(checkEqualsNumeric(2:11, seq(1, 10)))
(checkIdentical(iris[1:49, ], iris[iris$Species == "setosa",]))
(checkTrue(1 > 2))
(checkException(log(1)))
Log()	# See what's recorded in the log

# Create a test function and run it
foo <- function(x, y = 2) return(x * y)
test(foo) <- function () {
    #DEACTIVATED();
    checkEqualsNumeric(5, foo(2))
    checkEqualsNumeric(6, foo(2, 3))
    checkTrue(is.test(foo))
    checkTrue(is.test(test(foo)))
    checkIdentical(test(foo), attr(foo, "test"))
    checkException(foo("b"))
	checkException(foo(2, "a"))
}
(runTest(foo))

# Of course, everything is recorded in the log
Log()

clearLog()
}

\keyword{ utilities }
