% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/subscreencalc.R
\name{subscreencalc}
\alias{subscreencalc}
\title{Systematic screening of study data for subgroups}
\usage{
subscreencalc(data, eval_function, endpoints, treat = "trtp",
  subjectid = "subjid", factors = NULL, min_comb = 1, max_comb = 3,
  nkernel = 1, par_functions = "", verbose = T)
}
\arguments{
\item{data}{dataframe with study data}

\item{eval_function}{name of the function for data analysis}

\item{endpoints}{vector containing the names of the endpoint variables in data}

\item{treat}{name of variable in data that contains the treatment identfier, defaults to trtp}

\item{subjectid}{name of variable in data that contains the subject identifier, defaults to subjid}

\item{factors}{vector containg the names of variables that define the subgroups, defaults to NULL. If set to NULL, all variables in data are used that are not included in subjectid, treat, and endpoints}

\item{min_comb}{minimum number of factor combination levels to define subgroups, defaults to 1}

\item{max_comb}{maximum number of factor combination levels to define subgruops, defaults to 3}

\item{nkernel}{number of kernels for parallelization (defaults to 1)}

\item{par_functions}{vector of names of functions used in eval_function to be exported to cluster (needed only if nkernel > 1)}

\item{verbose}{switch on/off output of computational information}
}
\value{
an object of type SubScreenResult of the form
list(sge=H,
     max_comb=max_comb,
     min_comb=min_comb,
     subjectid=subjectid,
     endpoints=endpoints,
     treat=treat,
     factors=factors,
     results_total=eval_function(cbind(F,T)))
}
\description{
This function systematically calculates the defined outcome for every combination of subgroups
up to the given level (max_comb), i.e. the number of maximum combinations of subgroup defining factors.
If, e.g., in a study sex, age group (<=60, >60), BMI group (<=25, >25) are of interest, subgroups of level 2
would be, e.g, male subjects with BMI>25 or young females, while subgroups of level 3 would be any combination
of all three variables.
}
\details{
The evaluation function (eval_function) has to defined by the user. The result needs to be a vector of numerical values,
e.g., outcome variable(s) and number of observations/subjects. The input of eval_function is a data frame with the same
structure as the input data frame (data) used in the subsreencalc call. See example below.
Potential errors occurring due to small subgroups should be caught and handled within eval_function.
}
\examples{
# get the pbc data from the survival package
require(survival)
data(pbc, package="survival")

# generate categorical versions of some of the baseline covariates
pbc$ageg <- pbc$age <= median(pbc$age)
pbc$albuming <- pbc$albumin <= median(pbc$albumin)
pbc$phosg <- pbc$alk.phos <= median(pbc$alk.phos)
pbc$astg <- pbc$ast <= median(pbc$ast)
pbc$bilig <- pbc$bili <= median(pbc$bili)
pbc$cholg <- pbc$chol <= median(pbc$chol)
pbc$copperg <- pbc$copper <= median(pbc$copper)

# redefine censoring variable, consider transplant/death as event
pbc$event <- pbc$status
pbc$event[pbc$event==2] <- 1

pbcdat <- pbc[!is.na(pbc$trt), ]

# define function the eval_function()
hazardratio <- function(x) {
   hr <- tryCatch(exp(coxph(Surv(time, event) ~ trt, data=x)$coefficients[[1]]),
                 warning=function(w) {NA})
   N1 <- sum(x$trt==1)
   N2 <- sum(x$trt==2)

   data.frame(N1=N1, N2=N2, hr=hr)
 }

 # run subscreen

results <- subscreencalc(data=pbcdat,
                     eval_function=hazardratio,
                     endpoints = c("time", "event"),
                     treat="trt",
                     subjectid = "id",
                     factors=c("ageg", "sex", "bilig", "cholg", "copperg", "astg",
                               "albuming", "phosg"))

# visualize the results of the subgroup screening with a Shina app
\dontrun{subscreenshow(results, PreSelectTarge="hr", PreSelectXAxis="N1")}
}
\keyword{analysis}
\keyword{subgroup}
