% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sk_index.R
\name{sk_rescale}
\alias{sk_rescale}
\title{Up or down-scale a sk grid by an integer factor}
\usage{
sk_rescale(g, up = NULL, down = NULL)
}
\arguments{
\item{g}{a sk grid or any grid object accepted by \code{sk}}

\item{up}{integer > 0, or vector of two, the up-scaling factors}

\item{down}{integer > 0, or vector of two, the down-scaling factors}
}
\value{
a sk grid of the requested resolution
}
\description{
Changes the resolution of a sk grid by a factor of \code{up} or \code{down}. For down-scaling, this
introduces \code{NA}s at unobserved grid points (and does no interpolation).
}
\details{
Users should specify a sk grid \code{g} to re-scale and an integer scaling factor; either \code{up}
or \code{down} (and not both). This effects the scaling of resolution (\code{g[['gres']]}) by \code{up}
or \code{1/down}.

\code{up} (or \code{down}) should be a vector of two positive integers, the desired re-scaling
factors in the y and x dimensions, in that order, or a single value to be used for both.

When \code{up} is supplied, a lower resolution grid is returned comprising every \code{up}th grid
line of \code{g} along each dimension. All other grid lines, and any data values lying on them,
are ignored. \code{up} should be no greater than \code{dim(g) - 1}. Note that if \code{up} does not
evenly divide this number, the bounding box will shrink slightly.

When \code{down} is supplied, the function returns a higher resolution grid (say \code{g_fine}) with
the same bounding box as \code{g}. Along each dimension, every \code{down}th grid line of \code{g_fine}
coincides with a grid line of \code{g}. Any non-NA values found in \code{g[]} are copied to \code{g_fine},
and \code{g} can be recovered from \code{g_fine} with \code{sk_rescale(g_fine, up=down)}.
}
\examples{

# example data
gdim = c(50, 53)
pars = utils::modifyList(sk_pars(gdim), list(eps=1e-2))
g = sk_sim(gdim, pars)
plot(g)

# upscale
plot(sk_rescale(g, up=1)) # does nothing
plot(sk_rescale(g, up=2))

# downscale
sk_plot(sk_rescale(g, down=1)) # does nothing
sk_plot(sk_rescale(g, down=2))

# length-2 vectors to rescale differently in x and y directions
plot(sk_rescale(g, up=c(2,3)))
plot(sk_rescale(g, down=c(2,3)))

# invert a down-scaling
g_compare = sk_rescale(sk_rescale(g, down=c(5,3)), up=c(5,3))
all.equal(g, g_compare)

# multi-layer example with about 50\% of points missing
idx_miss = sample.int(length(g), round(0.5*length(g)))
g_multi = sk_sim(gdim, pars, n_layer=3)
g_multi[idx_miss,] = NA

# plot third layer, then down-scaled and up-scaled versions
sk_plot(g_multi, layer=3)
sk_plot(sk_rescale(g=g_multi, down=2), layer=3)
sk_plot(sk_rescale(g=g_multi, up=2), layer=3)

}
\seealso{
sk sk_cmean

Other indexing functions: 
\code{\link{sk_mat2vec}()},
\code{\link{sk_sub_find}()},
\code{\link{sk_sub_idx}()},
\code{\link{sk_vec2mat}()}

Other sk constructors: 
\code{\link{sk_snap}()},
\code{\link{sk_sub}()},
\code{\link{sk}()}
}
\concept{indexing functions}
\concept{sk constructors}
