% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dsmooth.R
\name{dsmooth}
\alias{dsmooth}
\title{Data-driven Local Polynomial for the Trend's Derivatives in Equidistant Time Series}
\usage{
dsmooth(
  y,
  d = c(1, 2),
  mu = c(0, 1, 2, 3),
  pp = c(1, 3),
  bStart.p = 0.15,
  bStart = 0.15
)
}
\arguments{
\item{y}{a numeric vector that contains the time series ordered from past to present.}

\item{d}{an integer 1 or 2 that defines the order of derivative.}

\item{mu}{an integer 0, ..., 3 that represents the smoothness parameter of
the kernel weighting function and thus defines the kernel function that will
be used within the local polynomial regression; is set to \emph{1} by
default.

\tabular{cl}{
\strong{Number} \tab \strong{Kernel}\cr
\emph{0} \tab Uniform Kernel\cr
\emph{1} \tab Epanechnikov Kernel\cr
\emph{2} \tab Bisquare Kernel\cr
\emph{3} \tab Triweight Kernel
}}

\item{pp}{an integer 1 (local linear regression) or 3 (local cubic
regression) that indicates the order of polynomial upon which c_[f], i.e. the
variance factor, will be calculated.}

\item{bStart.p}{a numeric object that indicates the starting value of the
bandwidth for the iterative process for the calculation of c_[f]; should be
0 < bStart.p < 0.5; is set to \emph{0.15} by default.}

\item{bStart}{a numeric object that indicates the starting value of the
bandwidth for the iterative process; should be 0 < bStart < 0.5; is set to
\emph{0.15} by default.}
}
\value{
The function returns a list with different components:

\describe{
\item{b0}{the optimal bandwidth chosen by the IPI-algorithm.}
\item{bStart}{the starting bandwidth for the local polynomial
regression based derivative estimation procedure; input argument.}
\item{bStart.p}{the starting bandwidth for the nonparametric trend estimation
that leads to the variance factor estimate; input argument.}
\item{cf0}{the estimated variance factor.}
\item{InfR}{the inflation rate setting.}
\item{iterations}{the bandwidths of the single iterations steps}
\item{mu}{the smoothness parameter of the second order kernel; input
argument.}
\item{n}{the number of observations.}
\item{niterations}{the total number of iterations until convergence.}
\item{orig}{the original input series; input argument.}
\item{p}{the order of polynomial for the local polynomial
regression used within derivative estimation procedure.}
\item{pp}{the order of polynomial for the local polynomial
regression used in the variance factor estimation; input argument.}
\item{res}{the estimated residual series.}
\item{ws}{the weighting systems used within the local polynomial regression;
only exists, if the final smoothing is done via a local polynomial
regression.}
\item{ye}{the nonparametric estimates of the derivative.}
}
}
\description{
This function runs through an iterative process in order to find the
 optimal bandwidth for the nonparametric estimation of the first or second
 derivative of the trend in an equidistant time series (with short-memory
 errors) and subsequently employs the obtained bandwidth via local
 polynomial regression.
}
\details{
The iterative-plug-in (IPI) algorithm, which numerically minimizes the
Asymptotic Mean Squared Error (AMISE), was proposed by Feng et al. (2019).

Define I[m^(k)] = int_[c_[b]]^[d_[b]] [m^(k)(x)]^2 dx,

beta_[v,k] =  int_[-1]^[1] u^k K(u)du and

R(K) = int_[-1]^[1] K^2(u)du.

The AMISE is then

AMISE(h) = h^(2(k-v)) * ( I[m^(k)]beta^2 / [k]^2 )
        + ( 2pi * c_[f](d_[b]-c_[b])R(K) / nh^(2v+1) )

with h being the bandwidth, k = p + 1 being the order of the equivalent
kernel, v being the order of derivative, 0 <= c_[b] < d_[b] <= 1, n being the
number of observations, c_[f] being the variance factor and K_[(v,k)](u)
being the k-th order equivalent kernel obtained for the estimation of m^[(v)]
in the interior. m^[(v)] is the v-th order derivative (v = 0, 1, 2, ...) of
the nonparametric trend.

The variance factor c_[f] is first obtained from a pilot-estimation of the
time series' nonparametric trend (v = 0) with polynomial order p_[p].
The estimate is then plugged into the iterative procedure for estimating
the first or second derivative (v = 1 or v = 2). For further details on the
asymptotic theory or the algorithm, we refer the user to Feng, Fritz and
Gries (2019) and Feng et al. (2019).

The function itself is applicable in the following way: Based on a data input
\emph{y}, an order of polynomial \emph{pp} for the variance factor estimation
procedure, a starting value for the relative bandwidth \emph{bStart.p} in the
variance factor estimation procedure, a kernel function defined by the
smoothness parameter \emph{mu} and a starting value for the relative
bandwidth \emph{bStart} in the bandwidth estimation procedure, an optimal
bandwidth is numerically calculated for the derivative of order \emph{d}.
In fact, aside from the input vector \emph{y}, every argument has a default
setting that can be adjusted for the individual case. However, it is
recommended to initially use the default values for the estimation of the
first derivative and adjust the argument \emph{d} to \emph{d = 2} for the
estimation of the second derivative. Following Feng, Gries and Fritz (2019),
the initial bandwidth does not affect the resulting optimal bandwidth in
theory. However in practice, local minima of the AMISE can influence the
results. Therefore, the default starting bandwidth is set to 0.15, the
suggested starting bandwidth by Feng, Gries and Fritz (2019) for the
data-driven  estimation of the first derivative. The recommended initial
bandwidth for the second derivative, however, is 0.2 and not 0.15. Thus, if
the algorithm does not give suitable results (especially for \emph{d = 2}),
the adjustment of the initial bandwidth might be a good starting point.
Analogously, the default starting bandwith for the trend estimation for the
variance factor is \emph{bStart.p = 0.15}, although according to Feng, Gries
and Fritz (2019), \emph{bStart.p = 0.1} is suggested for \emph{pp = 1} and
\emph{bStart.p = 0.2} for \emph{pp = 3}. The default is therefore a
compromise between the two suggested values. For more specific information
on the input arguments consult the section \emph{Arguments}.

After the bandwidth estimation, the nonparametric derivative of the series
is calulated with respect to the obtained optimal bandwidth by means of a
local polynomial regression. The output object is then a list that contains,
among other components, the original time series, the estimates of the
derivative and the estimated optimal bandwidth.

The default print method for this function delivers key numbers such as
the iteration steps and the generated optimal bandwidth rounded to the fourth
decimal. The exact numbers and results such as the estimated nonparametric
trend series are saved within the output object and can be addressed via the
\emph{$} sign.
}
\examples{
# Logarithm of test data
test_data <- gdpUS
y <- log(test_data$GDP)
t <- seq(from = 1947, to = 2019.25, by = 0.25)

# Applied dsmooth function for the trend's first derivative
result_d <- dsmooth(y, d = 1, mu = 1, pp = 1, bStart.p = 0.1, bStart = 0.15)
estim <- result_d$ye

# Plot of the results
plot(t, estim, xlab = "Year", ylab = "First derivative", type = "l",
 main = "Estimated first derivative of the trend for log-quarterly US-GDP, Q1 1947 - Q2 2019",
 cex.axis = 0.8, cex.main = 0.8, cex.lab = 0.8, bty = "n")

# Print result
result_d
}
\references{
Feng, Y., Gries, T. and Fritz, M. (2019). Data-driven
local polynomial for the trend and its derivatives in economic time
series. Discussion Paper. Paderborn University. (Not yet publshed)

Feng, Y., Gries, T., Letmathe, S. and Schulz, D. (2019). The smoots package
in R for semiparametric modeling of trend stationary time series. Discussion
Paper. Paderborn University. (Not yet published)
}
\author{
\itemize{
\item Yuanhua Feng (Department of Economics, Paderborn University), \cr
Author of the Algorithms \cr
Website: \url{https://wiwi.uni-paderborn.de/en/dep4/feng/}
\item Dominik Schulz (Student Assistant) (Department of Economics, Paderborn
University), \cr
Package Creator and Maintainer
}
}
