\name{rTranscriptData}
\alias{rTranscriptData}

\title{Simulation of artificial transcriptomic data}

\description{Simulates a Gaussian sample that mimics transcriptomic
 data, according to a given network, either steady-state or
 time-course data.  When several networks are given, multiple samples
 are generated.}

\usage{rTranscriptData(n,
                graph,
                ...,
                mu    = rep(0, p),
                sigma = 0.1)
}

\arguments{
 \item{n}{%
   integer or vector of integer indicating the sample sizes of each task
 }
 \item{graph}{
   a \code{simone.network} object typically generated either by
 \code{rNetwork} or \code{coNetwork}
 }
 \item{\dots}{%
   additional \code{simone.network} objects in case of multiple sample generation
 }
 \item{mu}{%
   if the network(s) is(are) directed, \code{mu} is the offset of the
   VAR(1) model that is used to generate the time-course data; if
   undirected, \code{mu} is the offset of the Gaussian vector.
 }
 \item{sigma}{%
   standard deviation of the noise term used in the simulation process
 }
}
\details{%
 If the network is directed, time-course data are simulated according
 to a VAR(1) model. If the network is undirected, steady-state data are
 generated by simulating an independent, identically distributed sample
 of a Gaussian vector.

 In both cases, samples are generated on the basis of \if{latex}{\out{$\mathbf{\Theta}$}}\if{html}{\out{&Theta;}}\if{text}{Theta}, as provided by \code{graph$Theta}.

 If the network is directed, samples are generated according to the
 following VAR(1) process:
\if{latex}{\deqn{
\left\{
 \begin{array}{l}
   X_0 \sim \mathcal{N}(0,\sigma) \\
   X_t \sim \mu + \mathbf{\Theta} X_{t-1} + \varepsilon_t \quad \textrm{for
all } t \in 1, \dots, n\\
   \varepsilon_t \sim \mathcal{N}(0,\sigma) \\
 \end{array} \right.}}

\if{html}{
\out{
<center>
<table>
      <tr>
         <td> X<sub>0</sub> follows N(0,&sigma;) </td>
      </tr>
      <tr>
         <td> X<sub>t</sub> = &mu; + &Theta; X<sub>t-1</sub> + &epsilon;<sub>t</sub>, for all t= 1,..., n</td>
      </tr>
      <tr>
	<td> &epsilon;<sub>t</sub> follows N(0,&sigma;).</td>
      </tr>
 </table>
</center>
}}
\if{text}{
X_0 follows N(0,sigma), for all t=1, ..., n, X_t = mu; + Theta X_{t-1} + epsilon_t, where epsilon_t follows N(0,sigma).
}

 If the network is undirected, samples are generated according to the
following Gaussian vector:
 \if{latex}{\deqn{\left\{ \begin{array}{l} %
   X_i \sim \mu + (\mathbf{\Theta}^{-1/2})^t U_i + \varepsilon_i \quad
\textrm{for all } i \in 1,\ldots,n \\
   U_i \sim \mathcal{N}(0,1) \\
   \varepsilon_i \sim \mathcal{N}(0,\sigma)\\
   \end{array}\right.}}
\if{html}{
\out{
<center>
<table>
      <tr>
         <td> X<sub>i</sub> = &mu; + t(&Theta;<sup>-1/2</sup>) U<sub>i</sub> + &epsilon;<sub>i</sub>, for all i in 1, ..., n,</td>
      </tr>
      <tr>
         <td> U<sub>i</sub> follows N(0,1) </td>
      </tr>
      <tr>
	<td> &epsilon;<sub>i</sub> follows N(0,&sigma;).</td>
      </tr>
 </table>
</center>
}}
\if{text}{
For all i=1, ..., n, X_i =  mu + t(Theta^-1/2) U_i + epsilon_i, where U_i follows N(0,1) and epsilon_t follows N(0,sigma).
}
 Numerically, \if{latex}{\eqn{\mathbf{\Theta}^{-1/2}}}\if{html}{\out{&Theta;}\out{<sup>-1/2</sup>}}\if{text}{Theta^-1/2} is computed with the
 Cholesky decomposition of the pseudo-inverse of \if{latex}{\eqn{\mathbf{\Theta}}}\if{html}{\out{&Theta;}}\if{text}{Theta}.
}

\value{
 Returns a list comprising :
 \item{X}{ matrix of simulated gene expression data, \code{n}
observations in rows, genes in columns}
 \item{tasks}{factor indicating the tasks corresponding to the
   simulated gene expression data in case of multiple networks.}
}
\author{J. Chiquet, C. Charbonnier}

\seealso{
  \code{\link{rNetwork}}, \code{\link{coNetwork}}.
}
\examples{
## time-Course data generation
##-----------------------------
# generate a directed network
n <- 20
p <- 5
g <- rNetwork(p, pi=5, directed=TRUE)
# Generate the data, data2 noisier than data1
data1  <- rTranscriptData(n,g)
data2  <- rTranscriptData(n,g,sigma=1)
matplot(1:n, data1$X,type= "l", xlab = "time points",
        ylab = "level of expression", col=rainbow(n,start=2/6,end = 3/6),
        ylim = range(c(data1$X,data2$X)),
        main="data2 (blue) generated with more noise than data1 (green)")
matlines(1:n,data2$X,type= "l",col = rainbow(n,start=4/6,end=5/6))

## steady-state data generation
##-----------------------------
# generate an undirected network
p <- 10
g <- rNetwork(p, pi=10)
data <- rTranscriptData(n=1000,g, sigma=0)
attach(data)
# Inference of Theta (here without dimension problems since p << n)
b <- sapply(1:p,function(x){
   tmp <- -solve(t(X[,-x]) \%*\% X[,-x]) \%*\% t(X[,-x]) \%*\% X[,x]
   res <- rep(NA,10)
   res[-x] <- tmp
   res[x] <- 1
   return(res)
  }
)
detach(data)
# comparison of theoretical Theta and inferred Theta
par(mfrow=c(1,2))
image(g$Theta, main = "Theoretical Theta")
image(b, main = "Inferred Theta")

## time-course multitask data generation
##--------------------------------------
# start by generating the networks
ancestor <- rNetwork(p=5, pi=5, name="ancestor", directed=TRUE)
child1   <- coNetwork(ancestor, 1, name = "child 1")
child2   <- coNetwork(ancestor, 1, name = "child 2")
# generate the data
n <- c(20,20)
data  <- rTranscriptData(n,child1,child2)
attach(data)
par(mfrow=c(2,1))
matplot(1:(n[1]),X[tasks ==1,],type= "l", main="Dataset from child 1",
        xlab = "time points", ylab = "level of expression")
matplot(1:(n[2]),X[tasks == 2,], type= "l", main="Dataset from child 2",
        xlab = "time points", ylab = "level of expression")
detach(data)
}
\keyword{datagen}