% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kinship_last_gen.R
\name{kinship_last_gen}
\alias{kinship_last_gen}
\title{Calculate kinship matrix for last generation of a pedigree with structured founders}
\usage{
kinship_last_gen(kinship, fam, ids, missing_vals = c("", 0))
}
\arguments{
\item{kinship}{The kinship matrix of the founders.
This matrix must have column and row names that identify each founder (matching codes in \code{fam$id}).
Individuals may be in a different order than \code{fam$id}.
Extra individuals in \code{kinship} but absent in \code{fam$id} will be silently ignored.
A traditional pedigree calculation would use \code{kinship = diag(n)/2} (plus appropriate column/row names), where \code{n} is the number of founders, to model unrelated and outbred founders.
However, if \code{kinship} measures the population kinship estimates between founders, the output is also a population kinship matrix (which combines the structural/ancestral and local/pedigree relatedness values into one).}

\item{fam}{The pedigree data.frame, in plink FAM format.
Only columns \code{id}, \code{pat}, and \code{mat} are required.
\code{id} must be unique and non-missing.
Founders must be present, and their \code{pat} and \code{mat} values must be missing (see below).
Non-founders must have both their parents be non-missing.
Parents must appear earlier than their children in the table.}

\item{ids}{A list containing vectors of IDs for each generation.
All these IDs must be present in \code{fam$id}.
If IDs in \code{fam} and \code{ids} do not fully agree, the code processes the IDs in the intersection, which is helpful when \code{fam} is pruned but \code{ids} is the original (larger) set.}

\item{missing_vals}{The list of ID values treated as missing.
\code{NA} is always treated as missing.
By default, the empty string ('') and zero (0) are also treated as missing (remove values from here if this is a problem).}
}
\value{
The kinship matrix of the last generation (the intersection of \code{ids[ length(ids) ]} and \code{fam$id}).
The columns/rows of this matrix are last-generation individuals in the order that they appear in \code{fam$id}.
}
\description{
A wrapper around the more general \code{\link[=kinship_fam]{kinship_fam()}}, specialized to save memory when only the last generation is desired (\code{\link[=kinship_fam]{kinship_fam()}} returns kinship for the entire pedigree in a single matrix).
This function assumes that generations are non-overlapping (met by the output of \code{\link[=sim_pedigree]{sim_pedigree()}}), in which case each generation \code{g} can be drawn from generation \code{g-1} data only.
That way, only two consecutive generations need be in memory at any given time.
The partitioning of individuals into generations is given by the \code{ids} parameter (again matches the output of \code{\link[=sim_pedigree]{sim_pedigree()}}).
}
\examples{
# A small pedigree, two parents and two children.
# A minimal fam table with the three required columns.
# Note "mother" and "father" have missing parent IDs, while children do not
library(tibble)
fam <- tibble(
  id = c('father', 'mother', 'child', 'sib'),
  pat = c(NA, NA, 'father', 'father'),
  mat = c(NA, NA, 'mother', 'mother')
)
# need an `ids` list separating the generations
ids <- list( c('father', 'mother'), c('child', 'sib') )

# Kinship of the parents, here two unrelated/outbred individuals:
kinship <- diag(2)/2
# Name the parents with same codes as in `fam`
# (order can be different)
colnames( kinship ) <- c('mother', 'father')
rownames( kinship ) <- c('mother', 'father')
# For a clearer example, make the father slightly inbred
# (a self-kinship value that exceeds 1/2):
kinship[2,2] <- 0.6

# calculate the kinship matrix of the children
kinship2 <- kinship_last_gen( kinship, fam, ids )
kinship2

}
\seealso{
Plink FAM format reference:
\url{https://www.cog-genomics.org/plink/1.9/formats#fam}
}
