#' @title Buttons Group checkbox Input Control
#'
#' @description
#' Create buttons grouped that act like checkboxes.
#'
#' @param inputId The \code{input} slot that will be used to access the value.
#' @param label Input label.
#' @param choices List of values to select from (if elements of the list are named
#'  then that name rather than the value is displayed to the user)
#' @param selected	The initially selected value.
#' @param status Color of the buttons
#' @param size Size of the buttons ('xs', 'sm', 'normal', 'lg')
#' @param direction Horizontal or vertical.
#' @param justified If TRUE, fill the width of the parent div.
#' @param individual If TRUE, buttons are separated.
#' @param checkIcon A list, if no empty must contain at least one element named 'yes'
#' corresponding to an icon to display if the button is checked.
#' @param width The width of the input, e.g. '400px', or '100\%'.
#'
#'
#' @return A buttons group control that can be added to a UI definition.
#'
#' @seealso \code{\link{updateCheckboxGroupButtons}}
#'
#' @examples
#' \dontrun{
#' ## Only run examples in interactive R sessions
#' if (interactive()) {
#'
#' ui <- fluidPage(
#'   checkboxGroupButtons(inputId = "somevalue",
#'                        label = "Make a choice: ",
#'                        choices = c("A", "B", "C")),
#'   verbatimTextOutput("value")
#' )
#' server <- function(input, output) {
#'   output$value <- renderText({ input$somevalue })
#' }
#' shinyApp(ui, server)
#' }
#' }
#'
#' @importFrom shiny restoreInput
#' @importFrom htmltools tags HTML validateCssUnit
#'
#' @export

checkboxGroupButtons <- function(
  inputId, label = NULL, choices, selected = NULL, status = "default", size = "normal",
  direction = "horizontal", justified = FALSE, individual = FALSE, checkIcon = list(),
  width = NULL
) {
  choices <- choicesWithNames(choices)
  selected <- shiny::restoreInput(id = inputId, default = selected)
  size <- match.arg(arg = size, choices = c("xs", "sm", "normal", "lg"))
  direction <- match.arg(arg = direction, choices = c("horizontal", "vertical"))
  status <- match.arg(arg = status, choices = c("default", "primary", "success", "info", "warning", "danger"))

  divClass <- if (individual) "" else "btn-group"
  if (!individual & direction == "vertical") {
    divClass <- paste0(divClass, "-vertical")
  }
  if (justified) {
    divClass <- paste(divClass, "btn-group-justified")
  }
  if (size != "normal") {
    divClass <- paste0(divClass, " btn-group-", size)
  }
  checkboxGroupButtonsTag <- htmltools::tags$div(
    class="form-group shiny-input-container shiny-input-checkboxgroup shiny-input-container-inline",
    style = if(!is.null(width)) paste("width:", htmltools::validateCssUnit(width)),
    if (!is.null(label)) htmltools::tags$label(class="control-label", `for`=inputId, label),
    if (!is.null(label)) htmltools::tags$br(),
    htmltools::tags$div(
      id=inputId, class="checkboxGroupButtons",
      style = "margin-top: 3px; margin-bottom: 3px;",
      style=if (justified) "width: 100%;",
      htmltools::tags$div(
        class=divClass, role="group", `aria-label`="...", `data-toggle`="buttons",
        class = "btn-group-container-sw",
        generateCBGB(inputId, choices, selected, status, size, checkIcon)
      )
    )
  )
  # Dep
  attachShinyWidgetsDep(checkboxGroupButtonsTag)
}





generateCBGB <- function(inputId, choices, selected, status, size, checkIcon) {
  btn_wrapper <- function(...) {
    htmltools::tags$div(
      class="btn-group",
      class=if (size != "normal") paste0("btn-group-", size),
      role="group",
      ...
    )
  }
  if (!is.null(checkIcon) && !is.null(checkIcon$yes)) {
    displayIcon <- TRUE
  } else {
    displayIcon <- FALSE
  }
  lapply(
    X = seq_along(choices),
    FUN = function(i) {
      btn_wrapper(
        htmltools::tags$button(
          class=paste0("btn checkbtn btn-", status),
          class=if (choices[i] %in% selected) "active",
          # if (checkIcon) tags$span(class="glyphicon glyphicon-ok"),
          if (displayIcon) htmltools::tags$span(class="check-btn-icon-yes", checkIcon$yes),
          if (displayIcon) htmltools::tags$span(class="check-btn-icon-no", checkIcon$no),
          htmltools::tags$input(
            type="checkbox", autocomplete="off",
            name=inputId, value=choices[i],
            checked=if (choices[i] %in% selected) "checked",
            htmltools::HTML(names(choices)[i])
          )
        )
      )
    }
  )
}




#' @title Change the value of a checkboxes group buttons input on the client
#'
#' @description
#' Change the value of a radio group buttons input on the client
#'
#' @param session The session object passed to function given to shinyServer.
#' @param inputId	The id of the input object.
#' @param selected The values selected.
#' @param label The label to set.
#' @param choices The new choices for the input.
#' @param status Status, only used if choices is not NULL.
#' @param size Size, only used if choices is not NULL.
#' @param checkIcon Icon, only used if choices is not NULL.
#'
#' @export
#'
#' @importFrom htmltools tagList
#'
#' @seealso \code{\link{checkboxGroupButtons}}
#'
#' @examples
#' \dontrun{
#' if (interactive()) {
#'
#' library(shiny)
#' library(shinyWidgets)
#'
#' # Example 1 ----
#'
#' ui <- fluidPage(
#'
#'   radioButtons(inputId = "up", label = "Update button :", choices = c("All", "None")),
#'
#'   checkboxGroupButtons(
#'     inputId = "btn", label = "Power :",
#'     choices = c("Nuclear", "Hydro", "Solar", "Wind"),
#'     selected = "Hydro"
#'   ),
#'
#'   verbatimTextOutput(outputId = "res")
#'
#' )
#'
#' server <- function(input,output, session){
#'
#'   observeEvent(input$up, {
#'     if (input$up == "All"){
#'       updateCheckboxGroupButtons(session, "btn", selected = c("Nuclear", "Hydro", "Solar", "Wind"))
#'     } else {
#'       updateCheckboxGroupButtons(session, "btn", selected = character(0))
#'     }
#'   }, ignoreInit = TRUE)
#'
#'   output$res <- renderPrint({
#'     input$btn
#'   })
#' }
#'
#' shinyApp(ui = ui, server = server)
#'
#'
#' # Example 2 ----
#'
#' library("shiny")
#' library("shinyWidgets")
#'
#' ui <- fluidPage(
#'   checkboxGroupButtons(
#'     inputId = "somevalue",
#'     choices = c("A", "B", "C"),
#'     label = "My label"
#'   ),
#'
#'   verbatimTextOutput(outputId = "res"),
#'
#'   actionButton(inputId = "updatechoices", label = "Random choices"),
#'   pickerInput(
#'     inputId = "updateselected", label = "Update selected:",
#'     choices = c("A", "B", "C"), multiple = TRUE
#'   ),
#'   textInput(inputId = "updatelabel", label = "Update label")
#' )
#'
#' server <- function(input, output, session) {
#'
#'   output$res <- renderPrint({
#'     input$somevalue
#'   })
#'
#'   observeEvent(input$updatechoices, {
#'     newchoices <- sample(letters, sample(2:6))
#'     updateCheckboxGroupButtons(
#'       session = session, inputId = "somevalue",
#'       choices = newchoices
#'     )
#'     updatePickerInput(
#'       session = session, inputId = "updateselected",
#'       choices = newchoices
#'     )
#'   })
#'
#'   observeEvent(input$updateselected, {
#'     updateCheckboxGroupButtons(
#'       session = session, inputId = "somevalue",
#'       selected = input$updateselected
#'     )
#'   }, ignoreNULL = TRUE, ignoreInit = TRUE)
#'
#'   observeEvent(input$updatelabel, {
#'     updateCheckboxGroupButtons(
#'       session = session, inputId = "somevalue",
#'       label = input$updatelabel
#'     )
#'   }, ignoreInit = TRUE)
#'
#' }
#'
#' shinyApp(ui = ui, server = server)
#'
#' }
#' }
updateCheckboxGroupButtons <- function(session, inputId, label = NULL, choices = NULL, selected = NULL,
                                       status = "default", size = "normal",
                                       checkIcon = list()) {
  if (!is.null(choices))
    choices <- choicesWithNames(choices)
  options <- if (!is.null(choices)) {
    format(htmltools::tagList(generateCBGB(inputId, choices, selected, status = status, size = size,
                                checkIcon = checkIcon)))
  }
  message <- dropNulls(list(selected = selected, options = options, label = label))
  session$sendInputMessage(inputId, message)
}

