#' Command bar item
#'
#' Helper function for constructing items for `CommandBar`.
#'
#' CommandBar expects items definition as a nested structure, which gets lengthy and verbose.
#' This function helps makes this definition shorter. Returns a list with all arguments passed through,
#' except for `text`, `icon` (which will inserted as proper `iconProps`) and `subitems` (which will be inserted as
#' proper `subMenuProps`).
#'
#' @param text Text to be displayed on the menu.
#' @param icon Optional name of an icon.
#' @param subitems Optional list of CommandBar items.
#' @param ... Additional props to pass to CommandBarItem.
#' @return Item suitable for use in the CommandBar.
#'
#' @seealso CommandBar
#' @export
CommandBarItem <- function(text, icon = NULL, subitems = NULL, ...) {
  props <- rlang::dots_list(...)

  props$text <- text
  if (is.character(icon)) props$iconProps <- list(iconName = icon)
  if (!is.null(subitems)) props$subMenuProps <- list(items = subitems)
  props
}

#' Basic Fluent UI page
#'
#' Creates a Fluent UI page with sensible defaults (included Fabric CSS classes, proper class given
#' to the body tag, suppressed Bootstrap).
#'
#' The Bootstrap library is suppressed by default,
#' as it doesn't work well with Fluent UI in general.
#'
#' @param ... The contents of the document body.
#' @param suppressBootstrap Whether to suppress Bootstrap.
#' @return Object which can be passed as the UI of a Shiny app.
#'
#' @export
fluentPage <- function(..., suppressBootstrap = TRUE) {
  htmltools::tags$body(class = "ms-Fabric",
    if (suppressBootstrap) htmltools::suppressDependencies("bootstrap"),
    htmltools::htmlDependency(
      name = "office-ui-fabric-core",
      version = "11.0.0",
      src = list(href = "https://static2.sharepointonline.com/files/fabric/office-ui-fabric-core/11.0.0/css/"),
      stylesheet = "fabric.min.css"
    ),
    ...
  )
}

#' parseTheme
#'
#' Reads a theme JSON generated by Theme Designer: \url{https://fabricweb.z5.web.core.windows.net/pr-deploy-site/refs/heads/master/theming-designer/}
#' and parses it to an object digestable by \code{\link{ThemeProvider}}
#'
#' @param path A path to JSON file containing the theme created in Theme Designer
#' @param json A JSON string containing the theme created in Theme Designer
#' @return A list with Fluent theme that can be used in \code{\link{ThemeProvider}}
#'
#' @seealso [ThemeProvider()] for usage of this function
#' @export
parseTheme <- function(path = NULL, json = NULL) {
  if (is.null(path) && is.null(json)) stop("Provide 'path' or 'json' argument")
  if (!is.null(path) && !is.null(json)) stop("Only one 'path' or 'json' argument must be used'")
  if (!is.null(path)) {
    palette <- jsonlite::read_json(path, simplifyVector = TRUE)
  } else {
    palette <- jsonlite::parse_json(json, simplifyVector = TRUE)
  }
  list(
    palette = palette
  )
}
