% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/emotion.R
\name{emotion}
\alias{emotion}
\title{Compute Emotion Rate}
\usage{
emotion(text.var, emotion_dt = lexicon::hash_nrc_emotions,
  valence_shifters_dt = lexicon::hash_valence_shifters,
  drop.unused.emotions = FALSE, un.as.negation = TRUE,
  un.as.negation.warn = isTRUE(all.equal(valence_shifters_dt,
  lexicon::hash_nrc_emotions)), n.before = 5, n.after = 2, ...)
}
\arguments{
\item{text.var}{The text variable.  Can be a \code{get_sentences} object or
a raw character vector though \code{get_sentences} is preferred as it avoids
the repeated cost of doing sentence boundary disambiguation every time
\code{sentiment} is run.}

\item{emotion_dt}{A \pkg{data.table} with a \code{token} and \code{emotion}
column (\code{tokens} are nested within the \code{emotion}s.  The table
cannot contain any duplicate rows and must have the \code{token} column set
as the key column (see \code{?data.table::setkey}).  The default emotion
table is \code{lexicon::hash_nrc_emotions}.}

\item{valence_shifters_dt}{A \pkg{data.table} of valence shifters that
can alter a polarized word's meaning and an integer key for negators (1),
amplifiers [intensifiers] (2), de-amplifiers [downtoners] (3) and adversative 
conjunctions (4) with x and y as column names.  For this purpose only 
negators is required/used.}

\item{drop.unused.emotions}{logical.  If \code{TRUE} unused/unfound emotion
levels will not be included in the output.}

\item{un.as.negation}{logical.  If \code{TRUE} then emotion words prefixed
with an 'un-' are treated as a negation.  For example,\code{"unhappy"} would 
be treated as \code{"not happy"}.  If an emotion word has an un- version in the
\code{emotion_dt} then no substitution is performed and an optional warning
will be given.}

\item{un.as.negation.warn}{logical.  If \code{TRUE} and if 
\code{un.as.negation} id \code{TRUE}, then a warning will be given if the 
-un version of an emotion term is already found within the \code{emotion_dt}.
Note that the default \code{emotion_dt}, \code{lexicon::hash_nrc_emotions}, 
will not give a warning unless it is explicitly set to do so.  There are
a number of emotion words in \code{lexicon::hash_nrc_emotions} that contain
un- prefixed versions already in the dictionary. Use:
\code{emotion('', un.as.negation.warn = TRUE)} to see these un- prefixed
emotion words that are contained within \code{lexicon::hash_nrc_emotions}.}

\item{n.before}{The number of words to consider as negated before
the emotion word.  To consider the entire beginning portion of a sentence
use \code{n.before = Inf}.  Note that a comma, colon, or semicolon acts as a 
boundary for considered words.  Only words between the emotion word and these
punctuation types will be considered.}

\item{n.after}{The number of words to consider as negated after
the emotion word.  To consider the entire ending portion of a sentence
use \code{n.after = Inf}.  Note that a comma, colon, or semicolon acts as a 
boundary for considered words.  Only words between the emotion word and these
punctuation types will be considered.}

\item{\ldots}{ignored.}
}
\value{
Returns a \pkg{data.table} of:
\itemize{
  \item element_id - The id number of the original vector passed to \code{emotion}
  \item sentence_id - The id number of the sentences within each \code{element_id}
  \item word_count - Word count
  \item emotion_type - Type designation from the \code{emotion} column of the \code{emotion_dt} table
  \item emotion_count - Count of the number of emotion words of that \code{emotion_type}
  \item emotion - A score of the percentage of emotion words of that \code{emotion_type}
}
}
\description{
Detect the rate of emotion at the sentence level.  This method uses a simple
dictionary lookup to find emotion words and then compute the rate per sentence.
The \code{emotion} score ranges between 0 (no emotion used) and 1 (all
words used were emotional).  Note that a single emotion phrase would count as 
just one in the \code{emotion_count} column but would count as two words in
the \code{word_count} column.
}
\examples{
mytext <- c(
    "I am not afraid of you",
    NA,
    "",
    "I love it [not really]", 
    "I'm not angry with you", 
    "I hate it when you lie to me.  It's so humiliating",
    "I'm not happpy anymore.  It's time to end it",
    "She's a darn good friend to me",
    "I went to the terrible store",
    "There is hate and love in each of us",
    "I'm no longer angry!  I'm really experiencing peace but not true joy.",
    
    paste("Out of the night that covers me, Black as the Pit from pole to", 
      "pole, I thank whatever gods may be For my unconquerable soul."
     ),
    paste("In the fell clutch of circumstance I have not winced nor cried",
        "aloud. Under the bludgeonings of chance My head is bloody, but unbowed."
    ),
    paste("Beyond this place of wrath and tears Looms but the Horror of the", 
        "shade, And yet the menace of the years Finds, and shall find, me unafraid."
    ),
    paste("It matters not how strait the gate, How charged with punishments", 
        "the scroll, I am the master of my fate: I am the captain of my soul."
    )    
    
)

## works on a character vector but not the preferred method avoiding the 
## repeated cost of doing sentence boundary disambiguation every time 
## `emotion` is run
emotion(mytext)

## preferred method avoiding paying the cost 
split_text <- get_sentences(mytext)
(emo <- emotion(split_text))
emotion(split_text, drop.unused.emotions = TRUE)

\dontrun{
plot(emo)
plot(emo, drop.unused.emotions = FALSE)
plot(emo, facet = FALSE)
plot(emo, facet = 'negated')

library(data.table)
fear <- emo[
    emotion_type == 'fear', ][, 
    text := unlist(split_text)][]
    
fear[emotion > 0,]

brady <- get_sentences(crowdflower_deflategate)
brady_emotion <- emotion(brady)
brady_emotion
}
}
\references{
Plutchik, R. (1962). The emotions: Facts and theories, and a new 
model. Random House studies in psychology. Random House.\cr\cr
Plutchik, R. (2001). The nature of emotions: Human emotions have deep 
evolutionary roots, a fact that may explain their complexity and provide tools 
for clinical practice. American Scientist , 89 (4), 344-350.
}
\seealso{
Other emotion functions: \code{\link{emotion_by}}
}
\concept{emotion functions}
\keyword{bad-words}
\keyword{cursing,}
\keyword{cussing,}
\keyword{emotion,}
\keyword{vulgarity,}
