% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/s2_list.R
\name{s2_list}
\alias{s2_list}
\title{Retrieve list of available S2 products.}
\usage{
s2_list(spatial_extent = NULL, tile = NULL, orbit = NULL,
  time_interval = c(Sys.Date() - 10, Sys.Date()), time_period = "full",
  level = "auto", apihub = NA, max_cloud = 100,
  output_type = "vector")
}
\arguments{
\item{spatial_extent}{A valid spatial object object of class \code{sf},
\code{sfc} or \code{sfg}}

\item{tile}{\code{string array} Sentinel-2 Tiles to be considered string (5-length character)}

\item{orbit}{\code{string array} Sentinel-2 orbit numbers to be considered}

\item{time_interval}{Dates to be considered, as a temporal vector (class \link{POSIXct} or
\link{Date}, or string in \code{YYYY-mm-dd} format) of length 1 (specific day) or 2 (time interval).}

\item{time_period}{(optional) Character:
\itemize{
\item "full" (default) means that all
the images included in the time window are considered;
\item "seasonal" means that only the single seasonal periods in the
window are used (i.e., with a time window from 2015-06-01 to
2017-08-31, the periods 2015-06-01 to 2015-08-31, 2016-06-01
to 2016-08-31 and 2017-06-01 to 2017-08-31 are considered).
}}

\item{level}{Character vector with one of the following:
- "auto" (default): check if level-2A is available on SciHub:
if so, list it; if not, list the corresponding level-1C
product
- "L1C": list available level-1C products
- "L2A": list available level-2A products}

\item{apihub}{Path of the "apihub.txt" file containing credentials
of SciHub account.
If NA (default), the default location inside the package will be used.}

\item{max_cloud}{Integer number (0-100) containing the maximum cloud
level of the tiles to be listed (default: no filter).}

\item{output_type}{Character: if 'vector' (default), the function returns
a vector or URLs, whose names are the SAFE names;
if 'data.table', the output is a data.table with metadata.}
}
\value{
Depending on the value of argument `output_type``,
a vector of available products (being each element an URL
and its name the product name), or a data.table with product metadata.
}
\description{
The function retrieves the list of available Sentinel-2
products satisfying given search criteria.
}
\note{
License: GPL 3.0
}
\examples{
\donttest{
pos <- sf::st_sfc(sf::st_point(c(9.85,45.81)), crs = 4326)
time_window <- as.Date(c("2016-05-01", "2017-07-30"))

# Full-period list
example_s2_list <- s2_list(
  spatial_extent = pos,
  tile = "32TNR",
  time_interval = time_window,
  orbit = "065"
)
print(example_s2_list)
# Print the dates of the retrieved products
as.vector(sort(sapply(names(example_s2_list), function(x) {
  strftime(safe_getMetadata(x,"nameinfo")$sensing_datetime)
})))

# Seasonal-period list
example_s2_list <- s2_list(
  spatial_extent = pos,
  tile = "32TNR",
  time_interval = time_window,
  time_period = "seasonal"
)
print(example_s2_list)
# Print the dates of the retrieved products
as.vector(sort(sapply(names(example_s2_list), function(x) {
  strftime(safe_getMetadata(x,"nameinfo")$sensing_datetime)
})))
}
}
\author{
Lorenzo Busetto, phD (2019) \email{lbusett@gmail.com} - Inspired by
function \code{getSentinel_query} of package \code{getSpatialData} by J. Schwalb-Willmann
(https://github.com/16EAGLE/getSpatialData)

Luigi Ranghetti, phD (2019) \email{luigi@ranghetti.info}
}
