% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rsvd.R
\name{rsvd}
\alias{rsvd}
\title{Randomized Singular Value Decomposition (rsvd).}
\usage{
rsvd(A, k = NULL, nu = NULL, nv = NULL, p = 10, q = 1,
  sdist = "unif", vt = FALSE)
}
\arguments{
\item{A}{array_like \cr
a real/complex input matrix (or data frame), with dimensions \eqn{(m, n)}.}

\item{k}{int, optional \cr
determines the target rank of the low-rank decomposition and should satisfy \eqn{k << min(m,n)}.}

\item{nu}{int, optional \cr
the number of left singular vectors to be computed. This must be between \eqn{0}
and \eqn{k}.}

\item{nv}{int, optional \cr
the number of right singular vectors to be computed. This must be between \eqn{0}
and \eqn{k}.}

\item{p}{int, optional \cr
oversampling parameter for (default \eqn{p=10}).}

\item{q}{int, optional \cr
number of power iterations (default \eqn{q=1}).}

\item{sdist}{str \eqn{c('normal', 'unif', 'col')}, optional \cr
Specifies the sampling distribution. \cr
\eqn{'unif'} : (default) Uniform `[-1,1]`. \cr
\eqn{'normal}' : Normal `~N(0,1)`. \cr
\eqn{'col'} : Random column sampling. \cr}

\item{vt}{bool (\eqn{TRUE}, \eqn{FALSE}), optional \cr
\eqn{TRUE} : returns the transposed right singular vectors \eqn{vt}. \cr
\eqn{FALSE} : (default) returns the right singular vectors as \eqn{v}, this is the format
as \code{\link{svd}} returns \eqn{v}.}

\item{.............}{.}
}
\value{
\code{rsvd} returns a list containing the following three components:
\item{d}{  array_like \cr
          Singular values; 1-d array of length \eqn{(k)}.
}

\item{u}{  array_like \cr
          Right singular values; array with dimensions \eqn{(m, k)}.
}

\item{v (or vt)}{  array_like \cr
          Left singular values; array with dimensions \eqn{(n, k)}. \cr
          Or if \eqn{vt=TRUE}, array with dimensions \eqn{(k, n)}.
}
\item{.............}{.}
}
\description{
Compute the near-optimal low-rank singular value decomposition (SVD) of a rectangular matrix.
}
\details{
The singular value decomposition (SVD) plays a central role in data analysis and scientific computing.
Randomized SVD (rSVD) is a fast algorithm to compute the approximate
low-rank SVD of a rectangular \eqn{(m,n)} matrix \eqn{A}
using a probablistic algorithm.
Given a target rank \eqn{k << n}, \code{rsvd} factors the input matrix \eqn{A} as
\eqn{A = U * diag(d) * V'}. The right singluar vectors are the columns of the
real or complex unitary matrix \eqn{U} . The left singular vectors are the columns
of the real or complex unitary matrix \eqn{V}. The singular values \eqn{d} are
non-negative and real numbers.

The parameter \eqn{p} is a oversampling parameter to improve the approximation.
A value between 5 and 10 is recommended and \eqn{p=10} is set by default.

The parameter \eqn{q} specifies the number of normalized power iterations
(subspace iterations) to reduce the approximation error. This is recommended
if the the singular values decay slowly. In practice 1 or 2 iterations
achieve good results, however, computing power iterations increases the
computational time. The number of power iterations is set to \eqn{q=1} by default.

If \eqn{k > (min(n,m)/1.5)}, a deterministic partial or truncated \code{\link{svd}}
algorithm might be faster.
}
\note{
The singular vectors are not unique and only defined up to sign
(a constant of modulus one in the complex case). If a left singular vector
has its sign changed, changing the sign of the corresponding right vector
gives an equivalent decomposition.
}
\examples{

# Create a n by n Hilbert matrix of order n,
# with entries H[i,j] = 1 / (i + j + 1).
hilbert <- function(n) { i <- 1:n; 1 / outer(i - 1, i, "+") }
H <- hilbert(n=50)

# Low-rank (k=10) matrix approximation using rsvd
k=10
s <- rsvd(H, k=k)
Hre <- s$u \%*\% diag(s$d) \%*\% t(s$v) # matrix approximation
print(100 * norm( H - Hre, 'F') / norm( H,'F')) # percentage error
# Compare to truncated base svd
s <- svd(H)
Hre <- s$u[,1:k] \%*\% diag(s$d[1:k]) \%*\% t(s$v[,1:k]) # matrix approximation
print(100 * norm( H - Hre, 'F') / norm( H,'F')) # percentage error

}
\author{
N. Benjamin Erichson, \email{nbe@st-andrews.ac.uk}
}
\references{
\itemize{
  \item  [1] N. Halko, P. Martinsson, and J. Tropp.
         "Finding structure with randomness: probabilistic
         algorithms for constructing approximate matrix
         decompositions" (2009).
         (available at arXiv \url{http://arxiv.org/abs/0909.4061}).
  \item  [2] S. Voronin and P.Martinsson.
         "RSVDPACK: Subroutines for computing partial singular value
         decompositions via randomized sampling on single core, multi core,
         and GPU architectures" (2015).
         (available at `arXiv \url{http://arxiv.org/abs/1502.05366}).
}
}
\seealso{
\code{\link{svd}}, \code{\link{rpca}}
}

