% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/opt.stiefel.R
\name{optStiefel}
\alias{optStiefel}
\title{Optimize a function on the Stiefel manifold}
\usage{
optStiefel(F, dF, Vinit, method = "bb", searchParams = NULL, tol = 1e-05,
  maxIters = 100, verbose = FALSE, maxLineSearchIters = 20)
}
\arguments{
\item{F}{A function V(P, S) -> \code{R^1}}

\item{dF}{A function to compute the gradient of F.  Returns a \code{P x S} matrix with \code{dF(X)_ij  = d(F(X))/dX_ij}}

\item{Vinit}{The starting point on the stiefel manifold for the optimization}

\item{method}{Line search type: "bb" or curvilinear}

\item{searchParams}{List of parameters for the line search algorithm.  If the line search algorithm is the standard curvilinear search than the search parameters are rho1 and rho2.  If the line search algorithm is "bb" then the parameters are rho and eta.}

\item{tol}{Convergence tolerance.  Optimization stops when Fprime < abs(tol), an approximate stationary point.}

\item{maxIters}{Maximum iterations for each gradient step}

\item{verbose}{Boolean indicating whether to print function value and iteration number at each step.}

\item{maxLineSearchIters}{Maximum iterations for for each line search (one step in the gradient descent algorithm)}
}
\value{
A stationary point of F on the Stiefel manifold.
}
\description{
Find a local optimum of a function defined on the stiefel manifold using algorithms described in Wen and Yin (2013).
}
\examples{
## Find the first eigenspace spanned by the first P eigenvectors for a 
## matrix M. The size of the matrix has been kept small and the tolerance 
## has been raised to keep the runtime 
## of this example below the CRAN submission threshold. 

N <- 500
P <- 3
Lam <- diag(c(10, 5, 3, rep(1, N-P)))
U <- rustiefel(N, N)
M <- U \%*\% Lam \%*\% t(U)

F <- function(V) { - sum(diag(t(V) \%*\% M \%*\% V)) }
dF <- function(V) { - 2*M \%*\% V }
V = optStiefel(F, dF, Vinit=rustiefel(N, P),
               method="curvilinear",
               searchParams=list(rho1=0.1, rho2=0.9, tau=1),tol=1e-4)
               
print(sprintf("Sum of first \%d eigenvalues is \%f", P, -F(V)))

}
\references{
(Wen and Yin, 2013)
}
\author{
Alexander Franks
}
