##' Assists creation of predicted value lines for values of a moderator variable.
##'
##' This is a "simple slope" plotter for linear regression.  The term
##' "simple slopes" was coined by psychologists (Aiken and West, 1991;
##' Cohen, et al 2002) to refer to analysis of interaction effects for
##' particular values of a moderating variable, be it continuous or
##' categorical. To use this function, the user should estimate a
##' regression (with as many variables as desired, including
##' interactions) and the resulting regression object is then supplied
##' to this function, along with user requests for plots of certain
##' variables.
##'
##' The variable \code{plotx} will be the horizontal plotting
##' variable; it must be numeric.  The variable \code{modx} is the
##' moderator variable. It may be either a numeric or a factor
##' variable.  A line will be drawn to represent the
##' predicted value for selected values of the moderator.
##'
##' The parameter \code{modxVals} is optional.  It is used to
##' fine-tune the values of the moderator that are used to create the
##' simple slope plot.  Numeric and factor moderators are treated
##' differently. If the moderator is a numeric variable, then some
##' particular values must be chosen for plottings. If the user does
##' not specify the parameter \code{modxVals}, then lines will be
##' drawn for the quantile values of the moderator.  If the moderator
##' is a factor, then lines are drawn for each different value of the
##' factor variable, unless the user specifies a subset of levels with
##' the \code{modxVals} parameter.
##'
##' For numeric moderators, the user may specify a vector of values
##' for the numeric moderator variable, such as c(1,2,3). The user may
##' also specify an algorithm, either "quantile" (which would be
##' selected by default) or "std.dev." The alternative method at this
##' time is "std.dev.", which causes 5 lines to be drawn. These lines
##' are the "standard deviations about the mean of \code{modx}" lines,
##' at which modx is set at mean - k* standard deviation, and k takes
##' on values -2, -1, 0, 1, 2.
##'
##' Here is a wrinkle. There can be many variables in a regression
##' model, and we are plotting only for the \code{plotx} and
##' \code{modx} variables. How should we calculate predicted values
##' when the values of the other variables are required?  For the
##' other variables, the ones that are not explicitly inlcluded in the
##' plot, we use the mean and mode, for numeric or factor variables
##' (respectively). Those values can be reviewed in the newdata object
##' that is created as a part of the output from this function
##'
##' @param model Required. Fitted regression object. Must have a predict method
##' @param plotx Required. String with name of IV to be plotted on x axis
##' @param modx  Required. String for moderator variable name. May be either numeric or factor.
##' @param modxVals Optional. If modx is numeric, either a character
##' string, "quantile", "std.dev.", or "table", or a vector of values
##' for which plotted lines are sought. If modx is a factor, the
##' default approach will create one line for each level, but the user
##' can supply a vector of levels if a subset is desired..
##' @param plotPoints TRUE or FALSE: Should the plot include the scatterplot points along with the lines.
##' @param plotLegend TRUE or FALSE: Include a default legend. Set to FALSE if use wants to run a different legend command after the plot has been drawn.
##' @param col An optional color vector.  By default, the R's builtin colors will be used,  which are "black", "red", and so forth.  Instead, a vector of color names can be supplied, as in c("pink","black", "gray70").  A color-vector generating function like rainbow(10) or gray.colors(5) can also be used. A vector of color names can be supplied with this function. Color names will be recycled if the plot requires more different colors than the user provides.
##' @param ... further arguments that are passed to plot
##' @export
##' @import car
##' @return The plot is drawn on the screen, and the return object includes the "newdata" object that was used to create the plot, along with the "modxVals" vector, the values of the moderator for which lines were drawn. It also includes the call that generated the plot.
##' @seealso plotCurves and testSlopes
##' @author Paul E. Johnson <pauljohn@@ku.edu>
##' @references
##' Aiken, L. S. and West, S.G. (1991). Multiple Regression: Testing and Interpreting Interactions. Newbury Park, Calif: Sage Publications.
##'
##' Cohen, J., Cohen, P., West, S. G., and Aiken, L. S. (2002). Applied Multiple Regression/Correlation Analysis for the Behavioral Sciences (Third.). Routledge Academic.
##' @example inst/examples/plotSlopes-ex.R

plotSlopes <-
  function (model = NULL, plotx = NULL, modx = NULL, modxVals = NULL,
            plotPoints = TRUE, plotLegend = TRUE, col, ...)
{
  if (is.null(model))
    stop("plotSlopes requires a fitted regression model.")
  if (is.null(plotx))
    stop("plotSlopes requires the name of the variable to be drawn on the x axis")
  if (is.null(modx))
    stop("plotSlopes requires the name of moderator variable for which several slopes are to be drawn")

  cutByTable <- function(x, n = 5) {
    table1 <- table(x)
    table1sort <-  sort(table1, decreasing = T)
    qs <- table1sort[1:n]
    names(qs) <- names(table1sort[1:n])
    invisible(qs)
  }

  cutByQuantile <- function(x){
    uniqueVals <- unique(x)
    if (length(uniqueVals) < 6) {
      qs <- cutByTable(x, 5)
      invisible(qs)
    } else {
      qs <- quantile(x, probs = c(0.25, 0.50, 0.75), na.rm = TRUE)
      invisible(qs)
    }
  }

  cutBySD <- function(x){
    uniqueVals <- unique(x)
    if (length(uniqueVals) < 6) {
      qs <- cutByTable(x, 5)
      invisible(qs)
    } else {
      mx <- round(mean(x, na.rm=T),2)
      sdx <- round(sd(x, na.rm=T),2)
      ##qs <- c(mx - 2*sdx, mx - sdx, mx, mx + sdx, mx + 2*sdx)
      ##suffix <- c("(m-2sd)","(m-sd)","(m)","(m+sd)","(m+2sd)")
      qs <- c(mx - sdx, mx, mx + sdx)
      suffix <- c("(m-sd)","(m)","(m+sd)")
      names(qs) <-  paste(qs, suffix)
      invisible(qs)
    }
  }


  cl <- match.call()
  mm <- model.matrix(model)
  depVar <- model$model[, 1]
  modxVar <- model$model[, modx]
  plotxVar <- model$model[, plotx]
  if (!is.numeric(plotxVar))
    stop(paste("plotSlopes: The variable", plotx, "should be a numeric variable"))
  ylab <- colnames(model$model)[1]
  plotyRange <- magRange(depVar, mult=c(1,1.2))
  plotxRange <- range(mm[, plotx], na.rm = TRUE)
  plotxSeq <- plotSeq(plotxRange, length.out = 40)

  if (is.factor(modxVar)) { ## modxVar is a factor
    if (is.null(modxVals)) {
      modxVals <- levels(modxVar)
    } else {
      if (!all(modxVals %in% levels(modxVar))) stop("modxVals includes non-observed levels of modxVar")
    }
  } else {                  ## modxVar is not a factor
    modxRange <- range(modxVar, na.rm=TRUE)
    if (is.null(modxVals)) {
      modxVals <- cutByQuantile(modxVar)
    } else {
      if (is.numeric(modxVals)) {
        ;# print("TODO: Insert some checks that modxVals are reasonable")
      } else {
        if (is.character(modxVals)) {
          modxVals <- match.arg(tolower(modxVals),
                                c("quantile", "std.dev."))
          print(modxVals)
          modxVals <- switch(modxVals,
                         table = cutByTable(modxVar),
                         quantile = cutByQuantile(modxVar),
                         "std.dev." = cutBySD(modxVar),
                         stop("unknown 'modxVals' algorithm"))
        }
      }
    }
  }
  lmx <- length(modxVals)
  if (missing(col)) col <- 1:lmx
  if (length(col) < lmx) rep(col, length.out = lmx)
  predictors <- colnames(model$model)[-1]
  predictors <- setdiff(predictors, c(modx, plotx))
  newdf <- data.frame(expand.grid(plotxRange, modxVals))
  colnames(newdf) <- c(plotx, modx)
  if (length(predictors) > 0) {
    newdf <- cbind(newdf, centralValues(as.data.frame(model$model[, predictors])))
    colnames(newdf) <- c(plotx, modx, predictors)
  }
  newdf$pred <- predict(model, newdata = newdf)
  dotargs <- list(...)
  if (!plotPoints){
    parms <- list(mm[, plotx], depVar, xlab = plotx, ylab = ylab,
         type = "n")
    parms <- modifyList(parms, dotargs)
    do.call("plot", parms)
  } else {
    if (is.factor(modxVar)) {
      parms <- list(mm[, plotx], depVar, xlab = plotx, ylab = ylab,
           col = col)
      parms <- modifyList(parms, dotargs)
      do.call("plot", parms)
    }
    else {
      parms <- list(mm[, plotx], depVar, xlab = plotx, ylab = ylab)
      parms <- modifyList(parms, dotargs)
      do.call("plot", parms)
    }
  }
  for (i in 1:lmx) {
    pdat <- newdf[newdf[, modx] %in% modxVals[i], ]
    lines(pdat[, plotx], pdat$pred, lty = i, col = col[i], lwd = 2)
  }
  if (is.null(names(modxVals))) {
    legnd <- paste(modx, " = ", modxVals, sep = "")
  }
  else {
    legnd <- paste(modx, " = ", names(modxVals), sep = "")
  }
  if(plotLegend) legend("topleft", legend = legnd, lty = 1:lmx, col = col,
         bg = "white", title= paste("moderator:", modx))

  invisible(list(call=cl, newdata=newdf, modxVals = modxVals))
}
