% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/riskParityPortfolio.R
\name{riskParityPortfolio}
\alias{riskParityPortfolio}
\title{Design of Risk Parity Portfolios}
\usage{
riskParityPortfolio(Sigma, b = NULL, mu = NULL, lmd_mu = 1e-04,
  lmd_var = 0, w_lb = 0, w_ub = 1,
  method_init = c("cyclical-spinu", "cyclical-roncalli", "newton"),
  method = c("sca", "alabama", "slsqp"), formulation = NULL,
  w0 = NULL, theta0 = NULL, gamma = 0.9, zeta = 1e-07,
  tau = NULL, maxiter = 50, ftol = 1e-08, wtol = 1e-06,
  use_gradient = TRUE)
}
\arguments{
\item{Sigma}{covariance or correlation matrix}

\item{b}{budget vector, aka risk budgeting targets. The default is the uniform
1/N vector.}

\item{mu}{vector of expected returns (only needed if the expected return term
is desired in the objective)}

\item{lmd_mu}{scalar that controls the importance of the expected return term}

\item{lmd_var}{scalar that controls the importance of the variance term
(only available for the SCA method for now).}

\item{w_lb}{lower bound on the value of each portfolio weight. If a vector,
then the lower bound is applied element-wise
(only available for the SCA method for now).}

\item{w_ub}{upper bound on the value of each portfolio weight. If a vector,
then the upper bound is applied element-wise
(only available for the SCA method for now).}

\item{method_init}{which algorithm to use for computing the initial portfolio
solution. We recommend choosing cyclical over Newton for high-dimensional
(N > 500) portfolios since it scales better in that regime. The
default is \code{"cyclical-spinu"}.}

\item{method}{which optimization method to use. The default is \code{"sca"}.}

\item{formulation}{string indicating the formulation to be used for the risk
parity optimization problem. It must be one of: \code{"diag", "rc-double-index",
"rc-over-b-double-index", "rc-over-var vs b", "rc-over-var",
"rc-over-sd vs b-times-sd", "rc vs b-times-var", "rc vs theta", or
"rc-over-b vs theta"}. If \code{formulation} is \code{NULL} and no additional terms
or constraints are set, such as expected return or shortselling, then
the vanilla risk parity portfolio will be returned. If formulation is
\code{"diag"} then the analytical solution of the risk parity optimization for
for a diagonal covariance matrix will be returned.}

\item{w0}{initial value for the portfolio weights. Default is the vanilla
portfolio computed either with cyclical or Newton methods.}

\item{theta0}{initial value for theta (in case formulation uses theta). If \code{NULL},
the optimum solution for a fixed vector of portfolio weights will be used.}

\item{gamma}{learning rate for the SCA method.}

\item{zeta}{factor used to decrease the learning rate at each iteration for the SCA method.}

\item{tau}{regularization factor. If \code{NULL}, a meaningful value will be used}

\item{maxiter}{maximum number of iterations for the SCA loop}

\item{ftol}{convergence tolerance on the risk contribution target}

\item{wtol}{convergence tolerance on the values of the portfolio weights}

\item{use_gradient}{(this parameter is meaningful only if method is either
\code{"alabama"} or \code{"slsqp"}) if \code{TRUE}, gradients of the objective function wrt
to the parameters will be used. This is strongly recommended to achieve faster results.}
}
\value{
a list containing possibly the following elements:
\item{\code{w}}{optimal portfolio vector}
\item{\code{risk_contribution}}{the risk contribution of every asset}
\item{\code{theta}}{the optimal value for theta (in case that it is part of
                    the chosen formulation)}
\item{\code{obj_fun}}{the sequence of values from the objective function at
                      each iteration}
\item{\code{risk_parity}}{the risk parity of the portfolio}
\item{\code{mean_return}}{the expected return of the portoflio if the mean
                          return term is included in the optimization}
\item{\code{variance}}{the variance of the portfolio if the variance term is
                       included in the optimization}
\item{\code{elapsed_time}}{elapsed time recorded at every iteration}
\item{\code{convergence}}{flag to indicate whether or not the optimization
converged. The value \code{TRUE} means it has converged and \code{FALSE} otherwise.}
}
\description{
This function designs risk-parity portfolios to equalize/distribute
the risk contributions of the different assets, which is missing if we simply
consider the overall volatility of the portfolio as in the mean-variance
Markowitz portfolio. In addition to the vanilla formulation, where the risk
contributions are perfectly equalized subject to no shortselling and budget
constraints, many other formulations are considered that allow for box
constraints, as well as the inclusion of additional objectives like the expected
return and overall variance. See the vignette for a detailed documentation and
comparison, with several illustrative examples.
}
\examples{
library(riskParityPortfolio)

# create covariance matrix
N <- 5
V <- matrix(rnorm(N^2), nrow = N)
Sigma <- cov(V)

# risk-parity portfolio
res <- riskParityPortfolio(Sigma)
names(res)
#> [1] "w"                 "risk_contribution"
res$w
#> [1] 0.04142886 0.38873465 0.34916787 0.09124019 0.12942842
res$risk_contribution
#> [1] 0.007361995 0.007361995 0.007361995 0.007361995 0.007361995
c(res$w * (Sigma \%*\% res$w))
#> [1] 0.007361995 0.007361995 0.007361995 0.007361995 0.007361995

# risk budggeting portfolio
res <- riskParityPortfolio(Sigma, b = c(0.4, 0.4, 0.1, 0.05, 0.05))
res$risk_contribution/sum(res$risk_contribution)
#> [1] 0.40 0.40 0.10 0.05 0.05

}
\references{
Y. Feng, and D. P. Palomar, "SCRIP: Successive Convex Optimization Methods
for Risk Parity Portfolio Design," \emph{IEEE Trans. on Signal Processing},
vol. 63, no. 19, pp. 5285-5300, Oct. 2015. (https://doi.org/10.1109/TSP.2015.2452219)

F. Spinu, "An Algorithm for Computing Risk Parity Weights," 2013.
Available at SSRN: https://ssrn.com/abstract=2297383 or http://dx.doi.org/10.2139/ssrn.2297383

T. Griveau-Billion, J. Richard, and T. Roncalli, "A fast algorithm for computing High-dimensional
risk parity portfolios," 2013. ArXiv preprint: https://arxiv.org/pdf/1311.4057.pdf
}
\author{
Ze Vinicius and Daniel P. Palomar
}
