% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kpca_poly.R
\name{step_kpca_poly}
\alias{step_kpca_poly}
\alias{tidy.step_kpca_poly}
\title{Polynomial Kernel PCA Signal Extraction}
\usage{
step_kpca_poly(
  recipe,
  ...,
  role = "predictor",
  trained = FALSE,
  num_comp = 5,
  res = NULL,
  degree = 2,
  scale_factor = 1,
  offset = 1,
  prefix = "kPC",
  keep_original_cols = FALSE,
  skip = FALSE,
  id = rand_id("kpca_poly")
)

\method{tidy}{step_kpca_poly}(x, ...)
}
\arguments{
\item{recipe}{A recipe object. The step will be added to the
sequence of operations for this recipe.}

\item{...}{One or more selector functions to choose which
variables will be used to compute the components. See
\code{\link[=selections]{selections()}} for more details. For the \code{tidy}
method, these are not currently used.}

\item{role}{For model terms created by this step, what analysis
role should they be assigned? By default, the function assumes
that the new principal component columns created by the original
variables will be used as predictors in a model.}

\item{trained}{A logical to indicate if the quantities for
preprocessing have been estimated.}

\item{num_comp}{The number of PCA components to retain as new
predictors. If \code{num_comp} is greater than the number of columns
or the number of possible components, a smaller value will be
used.}

\item{res}{An S4 \code{\link[kernlab:kpca]{kernlab::kpca()}} object is stored
here once this preprocessing step has be trained by
\code{\link[=prep.recipe]{prep.recipe()}}.}

\item{degree, scale_factor, offset}{Numeric values for the polynomial kernel function.}

\item{prefix}{A character string that will be the prefix to the
resulting new variables. See notes below.}

\item{keep_original_cols}{A logical to keep the original variables in the
output. Defaults to \code{FALSE}.}

\item{skip}{A logical. Should the step be skipped when the
recipe is baked by \code{\link[=bake.recipe]{bake.recipe()}}? While all operations are baked
when \code{\link[=prep.recipe]{prep.recipe()}} is run, some operations may not be able to be
conducted on new data (e.g. processing the outcome variable(s)).
Care should be taken when using \code{skip = TRUE} as it may affect
the computations for subsequent operations}

\item{id}{A character string that is unique to this step to identify it.}

\item{x}{A \code{step_kpca_poly} object}
}
\value{
An updated version of \code{recipe} with the new step
added to the sequence of existing steps (if any). For the
\code{tidy} method, a tibble with columns \code{terms} (the
selectors or variables selected).
}
\description{
\code{step_kpca_poly} a \emph{specification} of a recipe step that
will convert numeric data into one or more principal components
using a polynomial kernel basis expansion.
}
\details{
Kernel principal component analysis (kPCA) is an
extension of a PCA analysis that conducts the calculations in a
broader dimensionality defined by a kernel function. For
example, if a quadratic kernel function were used, each variable
would be represented by its original values as well as its
square. This nonlinear mapping is used during the PCA analysis
and can potentially help find better representations of the
original data.

This step requires the \pkg{dimRed} and \pkg{kernlab} packages.
If not installed, the step will stop with a note about installing
these packages.

As with ordinary PCA, it is important to standardize the
variables prior to running PCA (\code{step_center} and
\code{step_scale} can be used for this purpose).

The argument \code{num_comp} controls the number of components that
will be retained (the original variables that are used to derive
the components are removed from the data). The new components
will have names that begin with \code{prefix} and a sequence of
numbers. The variable names are padded with zeros. For example,
if \code{num_comp < 10}, their names will be \code{kPC1} -
\code{kPC9}. If \code{num_comp = 101}, the names would be
\code{kPC001} - \code{kPC101}.
}
\examples{
library(modeldata)
data(biomass)

biomass_tr <- biomass[biomass$dataset == "Training",]
biomass_te <- biomass[biomass$dataset == "Testing",]

rec <- recipe(HHV ~ carbon + hydrogen + oxygen + nitrogen + sulfur,
              data = biomass_tr)

kpca_trans <- rec \%>\%
  step_YeoJohnson(all_numeric_predictors()) \%>\%
  step_normalize(all_numeric_predictors()) \%>\%
  step_kpca_poly(all_numeric_predictors())

if (require(dimRed) & require(kernlab)) {
  kpca_estimates <- prep(kpca_trans, training = biomass_tr)

  kpca_te <- bake(kpca_estimates, biomass_te)

  rng <- extendrange(c(kpca_te$kPC1, kpca_te$kPC2))
  plot(kpca_te$kPC1, kpca_te$kPC2,
       xlim = rng, ylim = rng)

  tidy(kpca_trans, number = 3)
  tidy(kpca_estimates, number = 3)
}
}
\references{
Scholkopf, B., Smola, A., and Muller, K. (1997).
Kernel principal component analysis. \emph{Lecture Notes in
Computer Science}, 1327, 583-588.

Karatzoglou, K., Smola, A., Hornik, K., and Zeileis, A. (2004).
kernlab - An S4 package for kernel methods in R. \emph{Journal
of Statistical Software}, 11(1), 1-20.
}
\seealso{
\code{\link[=step_pca]{step_pca()}} \code{\link[=step_ica]{step_ica()}}
\code{\link[=step_isomap]{step_isomap()}} \code{\link[=recipe]{recipe()}} \code{\link[=prep.recipe]{prep.recipe()}}
\code{\link[=bake.recipe]{bake.recipe()}}
}
\concept{basis_expansion}
\concept{kernel_methods}
\concept{pca}
\concept{preprocessing}
\concept{projection_methods}
\keyword{datagen}
