% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/expand_distances_each.R
\name{expand_distances_each}
\alias{expand_distances_each}
\title{Expand the distances to an origin in each dimension}
\usage{
expand_distances_each(
  data,
  cols = NULL,
  multipliers = NULL,
  multipliers_fn = NULL,
  origin = NULL,
  origin_fn = NULL,
  exponentiate = FALSE,
  add_one_exp = TRUE,
  suffix = "_expanded",
  keep_original = TRUE,
  mult_col_name = ifelse(isTRUE(exponentiate), ".exponents", ".multipliers"),
  origin_col_name = ".origin",
  overwrite = FALSE
)
}
\arguments{
\item{data}{\code{data.frame} or \code{vector}.}

\item{cols}{Names of columns in \code{`data`} to expand.
Each column is considered a dimension to expand in.}

\item{multipliers}{Constant(s) to multiply/exponentiate the distance to the origin by.
A scalar to use in all dimensions or
a \code{vector} with one scalar per dimension.

\strong{N.B.} When \code{`exponentiate`} is \code{TRUE}, the \code{`multipliers`} become \emph{exponents}.}

\item{multipliers_fn}{Function for finding the \code{`multipliers`}.

\strong{Input}: Each column will be passed as a \code{vector} in the order of \code{`cols`}.

\strong{Output}: A \code{numeric vector} with one element per dimension.

Just as for \code{`origin_fn`}, it can be created with
\code{\link[rearrr:create_origin_fn]{create_origin_fn()}} if you want to apply
the same function to each dimension. See \code{`origin_fn`}.}

\item{origin}{Coordinates of the origin to expand around.
A scalar to use in all dimensions
or a \code{vector} with one scalar per dimension.

\strong{N.B.} Ignored when \code{`origin_fn`} is not \code{NULL}.}

\item{origin_fn}{Function for finding the origin coordinates.

\strong{Input}: Each column will be passed as a \code{vector} in the order of \code{`cols`}.

\strong{Output}: A \code{vector} with one scalar per dimension.

Can be created with \code{\link[rearrr:create_origin_fn]{create_origin_fn()}} if you want to apply
the same function to each dimension.

E.g. \code{`create_origin_fn(median)`} would find the median of each column.

\strong{Built-in functions} are \code{\link[rearrr:centroid]{centroid()}},
\code{\link[rearrr:most_centered]{most_centered()}},
and \code{\link[rearrr:midrange]{midrange()}}}

\item{exponentiate}{Whether to exponentiate instead of multiplying. (Logical)}

\item{add_one_exp}{Whether to add the \code{sign} (either \code{1} or \code{-1})
before exponentiating to ensure the values don't contract.
The added value is subtracted after the exponentiation. (Logical)

Exponentiation becomes:

\code{x <- x + sign(x)}

\code{x <- sign(x) * abs(x) ^ multiplier}

\code{x <- x - sign(x)}

\strong{N.B.} Ignored when \code{`exponentiate`} is \code{FALSE}.}

\item{suffix}{Suffix to add to the names of the generated columns.

Use an empty string (i.e. \code{""}) to overwrite the original columns.}

\item{keep_original}{Whether to keep the original columns. (Logical)

Some columns may have been overwritten, in which case only the newest versions are returned.}

\item{mult_col_name}{Name of new column with the multiplier(s). If \code{NULL}, no column is added.}

\item{origin_col_name}{Name of new column with the origin coordinates. If \code{NULL}, no column is added.}

\item{overwrite}{Whether to allow overwriting of existing columns. (Logical)}
}
\value{
\code{data.frame} (\code{tibble}) with the expanded columns,
along with the applied multiplier/exponent and origin coordinates.
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

Moves the data points in n-dimensional space such that their distance
to the specified origin is increased/decreased \emph{in each dimension separately}.
A \code{`multiplier`} greater than 1 leads to expansion,
while a positive \code{`multiplier`} lower than 1 leads to contraction.

The origin can be supplied as coordinates or as a function that returns coordinates. The
latter can be useful when supplying a grouped \code{data.frame} and expanding around e.g. the centroid
of each group.

The multipliers/exponents can be supplied as constant(s) or as a function that returns constants.
The latter can be useful when supplying a grouped \code{data.frame} and the multiplier/exponent depends
on the data in the groups.
If supplying multiple constants, there must be one per dimension (length of \code{`cols`}).

For expansion of the \emph{multidimensional} distance, use \code{\link[rearrr:expand_distances_each]{expand_distances()}}.

\strong{NOTE}: When exponentiating, the default is to first add \code{1} or \code{-1}
(depending on the sign of the distance) to the distances,
to ensure expansion even when the distance is between \code{-1} and \code{1}.
If you need the purely exponentiated distances,
disable \code{`add_one_exp`}.
}
\details{
For each value of each dimension (column), either multiply or exponentiate by the multiplier:

\code{# Multiplication}

\code{x <- x * multiplier}

\code{# Exponentiation}

\code{x <- sign(x) * abs(x) ^ multiplier}

Note: By default (when \code{`add_one_exp`} is \code{TRUE}),
we add the sign (\code{1 / -1}) of the value before the exponentiation
and subtract it afterwards. See \code{`add_one_exp`}.
}
\examples{
# Attach packages
library(rearrr)
library(dplyr)
library(purrr)
library(ggplot2)

# Set seed
set.seed(1)

# Create a data frame
df <- data.frame(
  "x" = runif(20),
  "y" = runif(20),
  "g" = rep(1:4, each = 5)
)

# Expand values in the two dimensions (x and y)
# With the origin at x=0.5, y=0.5
# We expand x by 2 and y by 4
expand_distances_each(
  data = df,
  cols = c("x", "y"),
  multipliers = c(2, 4),
  origin = c(0.5, 0.5)
)

# Expand values in the two dimensions (x and y)
# With the origin at x=0.5, y=0.5
# We expand both by 3
expand_distances_each(
  data = df,
  cols = c("x", "y"),
  multipliers = 3,
  origin = 0.5
)

# Expand values in one dimension (x)
# With the origin at x=0.5
# We expand by 3
expand_distances_each(
  data = df,
  cols = c("x"),
  multipliers = 3,
  origin = 0.5
)

# Expand x and y around the centroid
# We use exponentiation for a more drastic effect
# The add_one_exp makes sure it expands
# even when x or y is in the range [>-1, <1]
# To compare multiple exponents, we wrap the
# call in purrr::map_dfr
df_expanded <- purrr::map_dfr(
  .x = c(1, 2.0, 3.0, 4.0),
  .f = function(exponent) {
    expand_distances_each(
      data = df,
      cols = c("x", "y"),
      multipliers = exponent,
      origin_fn = centroid,
      exponentiate = TRUE,
      add_one_exp = TRUE
    )
  }
)
df_expanded

# Plot the expansions of x and y around the overall centroid
ggplot(df_expanded, aes(x = x_expanded, y = y_expanded, color = factor(.exponents))) +
  geom_vline(
    xintercept = df_expanded[[".origin"]][[1]][[1]],
    size = 0.2, alpha = .4, linetype = "dashed"
  ) +
  geom_hline(
    yintercept = df_expanded[[".origin"]][[1]][[2]],
    size = 0.2, alpha = .4, linetype = "dashed"
  ) +
  geom_point() +
  theme_minimal() +
  labs(x = "x", y = "y", color = "Exponent")

# Expand x and y around the centroid using multiplication
# To compare multiple multipliers, we wrap the
# call in purrr::map_dfr
df_expanded <- purrr::map_dfr(
  .x = c(1, 2.0, 3.0, 4.0),
  .f = function(multiplier) {
    expand_distances_each(df,
      cols = c("x", "y"),
      multipliers = multiplier,
      origin_fn = centroid,
      exponentiate = FALSE
    )
  }
)
df_expanded

# Plot the expansions of x and y around the overall centroid
ggplot(df_expanded, aes(x = x_expanded, y = y_expanded, color = factor(.multipliers))) +
  geom_vline(
    xintercept = df_expanded[[".origin"]][[1]][[1]],
    size = 0.2, alpha = .4, linetype = "dashed"
  ) +
  geom_hline(
    yintercept = df_expanded[[".origin"]][[1]][[2]],
    size = 0.2, alpha = .4, linetype = "dashed"
  ) +
  geom_point() +
  theme_minimal() +
  labs(x = "x", y = "y", color = "Multiplier")

# Expand x and y with different multipliers
# around the centroid using multiplication
df_expanded <- expand_distances_each(
  df,
  cols = c("x", "y"),
  multipliers = c(1.25, 10),
  origin_fn = centroid,
  exponentiate = FALSE
)
df_expanded

# Plot the expansions of x and y around the overall centroid
# Note how the y axis is expanded a lot more than the x-axis
ggplot(df_expanded, aes(x = x_expanded, y = y_expanded)) +
  geom_vline(
    xintercept = df_expanded[[".origin"]][[1]][[1]],
    size = 0.2, alpha = .4, linetype = "dashed"
  ) +
  geom_hline(
    yintercept = df_expanded[[".origin"]][[1]][[2]],
    size = 0.2, alpha = .4, linetype = "dashed"
  ) +
  geom_line(aes(color = "Expanded")) +
  geom_point(aes(color = "Expanded")) +
  geom_line(aes(x = x, y = y, color = "Original")) +
  geom_point(aes(x = x, y = y, color = "Original")) +
  theme_minimal() +
  labs(x = "x", y = "y", color = "Multiplier")

#
# Contraction
#

# Group-wise contraction to create clusters
df_contracted <- df \%>\%
  dplyr::group_by(g) \%>\%
  expand_distances_each(
    cols = c("x", "y"),
    multipliers = 0.07,
    suffix = "_contracted",
    origin_fn = centroid
  )

# Plot the clustered data point on top of the original data points
ggplot(df_contracted, aes(x = x_contracted, y = y_contracted, color = factor(g))) +
  geom_point(aes(x = x, y = y, color = factor(g)), alpha = 0.3, shape = 16) +
  geom_point() +
  theme_minimal() +
  labs(x = "x", y = "y", color = "g")
}
\seealso{
Other mutate functions: 
\code{\link{apply_transformation_matrix}()},
\code{\link{cluster_groups}()},
\code{\link{dim_values}()},
\code{\link{expand_distances}()},
\code{\link{flip_values}()},
\code{\link{roll_values}()},
\code{\link{rotate_2d}()},
\code{\link{rotate_3d}()},
\code{\link{shear_2d}()},
\code{\link{shear_3d}()},
\code{\link{swirl_2d}()},
\code{\link{swirl_3d}()}

Other expander functions: 
\code{\link{expand_distances}()}

Other distance functions: 
\code{\link{closest_to}()},
\code{\link{dim_values}()},
\code{\link{distance}()},
\code{\link{expand_distances}()},
\code{\link{furthest_from}()},
\code{\link{swirl_2d}()},
\code{\link{swirl_3d}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{distance functions}
\concept{expander functions}
\concept{mutate functions}
