% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lcp.R
\name{find_lcp}
\alias{find_lcp}
\alias{find_lcp,LcpFinder,numeric-method}
\title{Find the LCP between two points on a \code{Quadtree}}
\usage{
\S4method{find_lcp}{LcpFinder,numeric}(x, y, use_original_end_points = FALSE)
}
\arguments{
\item{x}{a \code{\link{LcpFinder}}}

\item{y}{two-element numeric vector; the x and y coordinates of the the
destination point}

\item{use_original_end_points}{boolean; by default the start and end points
of the returned path are not the points given by the user but instead the
centroids of the cells that those points fall in. If this parameter is set
to \code{TRUE} the start and end points (representing the cell centroids)
are replaced with the actual points specified by the user. Note that this
is done after the calculation and has no effect on the path found by the
algorithm.}
}
\value{
Returns a five column matrix representing the LCP. It has the
  following columns:
  \itemize{
     \item{\code{x}: }{x coordinate of this point (centroid of the cell)}
     \item{\code{y}: }{y coordinate of this point (centroid of the cell)}
     \item{\code{cost_tot}: }{the cumulative cost up to this point}
     \item{\code{dist_tot}: }{the cumulative distance up to this point - note
     that this is not straight-line distance, but instead the distance along
     the path}
     \item{\code{cost_cell}: }{the cost of the cell that contains this point}
   }

  If no path is possible between the two points, a zero-row matrix with the
  previously described columns is returned.

  \strong{Important note:} the \code{use_original_end_points} option
  \strong{only} changes the x and y coordinates of the first and last points
  - it doesn't change the \code{cost_tot} or \code{dist_tot} columns. This
  means that even though the start and end points have changed, the
  \code{cost_tot} and \code{dist_tot} columns still represent the cost and
  distance using the cell centroids of the start and end cells.
}
\description{
Finds the least-cost path (LCP) from the start point (the point
  used to create the \code{\link{LcpFinder}}) to another point, using a
  \code{\link{Quadtree}} as a resistance surface.
}
\details{
See the vignette 'quadtree-lcp' for more details and examples (i.e. run
  \code{vignette("quadtree-lcp", package = "quadtree")})
}
\examples{
####### NOTE #######
# see the "quadtree-lcp" vignette for more details and examples:
# vignette("quadtree-lcp", package = "quadtree")
####################

library(quadtree)
data(habitat)

# create a quadtree
qt <- quadtree(habitat, split_threshold = .1, adj_type = "expand")
plot(qt, crop = TRUE, na_col = NULL, border_lwd = .4)

# define our start and end points
start_pt <- c(6989, 34007)
end_pt <- c(33015, 38162)

# create the LCP finder object and find the LCP
lcpf <- lcp_finder(qt, start_pt)
path <- find_lcp(lcpf, end_pt)

# plot the LCP
plot(qt, crop = TRUE, na_col = NULL, border_col = "gray30", border_lwd = .4)
points(rbind(start_pt, end_pt), pch = 16, col = "red")
lines(path[, 1:2], col = "black")
}
\seealso{
\code{\link{lcp_finder}()} creates the LCP finder object used as
  input to this function. \code{\link{find_lcps}()} calculates all LCPs
  whose cost-distance is less than some value. \code{\link{summarize_lcps}()}
  outputs a summary matrix of all LCPs that have been calculated so far.
}
