use_envname <- function(
    envname = NULL,
    method = "spark_connect",
    version = NULL,
    messages = FALSE,
    match_first = FALSE,
    ignore_reticulate_python = FALSE,
    ask_if_not_installed = interactive()) {
  ret_python <- reticulate_python_check(ignore_reticulate_python)

  if (ret_python != "") {
    return(set_names(ret_python, "env_var"))
  }

  if (!is.null(envname)) {
    return(set_names(envname, "argument"))
  }

  if (is.null(version)) {
    cli_abort("A cluster {.code version} is required, please provide one")
  }

  if (method == "spark_connect") {
    env_base <- "r-sparklyr-pyspark-"
    run_code <- glue("pysparklyr::install_pyspark(version = \"{version}\")")
  } else {
    env_base <- "r-sparklyr-databricks-"
    run_code <- glue("pysparklyr::install_databricks(version = \"{version}\")")
  }

  con_label <- connection_label(method)
  run_full <- "{.header Run: {.run {run_code}} to install.}"

  sp_version <- version_prep(version)
  envname <- as.character(glue("{env_base}{sp_version}"))
  envs <- find_environments(env_base)

  match_one <- length(envs) > 0
  match_exact <- length(envs[envs == envname]) > 0

  # There were 0 environments found
  if (!match_one && !match_exact) {
    ret <- set_names(envname, "unavailable")
    msg_1 <- paste0(
      "No {.emph viable} Python Environment was identified for ",
      "{.emph {con_label}} version {.emph {version}}"
    )
    msg_2 <- NULL
  }

  # Found an exact match
  if (match_one && match_exact) {
    ret <- set_names(envname, "exact")
    msg_1 <- paste0(
      "No {.emph matching} Python Environment was found for ",
      "{.emph {con_label}} version {.emph {version}}"
    )
    msg_2 <- NULL
  }

  # There are environments, but no exact match, and argument says
  # to choose the most recent environment
  if (match_one && !match_exact && match_first) {
    ret <- set_names(envs[1], "first")
    msg_1 <- paste0(
      "No {.emph exact} Python Environment was found for ",
      "{.emph {con_label}} version {.emph {version}}. \n"
    )
    msg_2 <- paste0(
      "{.header If the exact version is not installed, {.code sparklyr} will ",
      "use {.code {ret}}}"
    )
  }

  # There are environments, but no exact match
  if (match_one && !match_exact && !match_first) {
    msg_1 <- paste0(
      "No {.emph exact} Python Environment was found for ",
      "{.emph {con_label}} version {.emph {version}}"
    )
    msg_2 <- "{.header The default Python environment may not work correctly}"
    ret <- set_names(envname, "unavailable")
  }

  ret_name <- names(ret)
  if (messages && ret_name != "exact") {
    cli_div(theme = cli_colors())
    if (ask_if_not_installed) {
      cli_alert_warning(msg_1)
      cli_bullets(c(
        " " = msg_2,
        " " = "Do you wish to install {con_label} version {version}?"
      ))
      choice <- utils::menu(choices = c("Yes", "No", "Cancel"))
      if (choice == 1) {
        ret <- set_names(envname, "prompt")
        if (method == "databricks_connect") {
          install_databricks(version = version, as_job = FALSE)
        }
        if (method == "spark_connect") {
          install_pyspark(version = version, as_job = FALSE)
        }
      }
      if (choice == 2) {
        ret <- set_names(ret, "prompt")
      }
      if (choice == 3) {
        cli_abort("Operation cancelled by user")
      }
    } else {
      if (ret_name == "unavailable") {
        cli_abort(c(msg_1, msg_2, run_full))
      }
      if (ret_name == "first") {
        cli_alert_warning(msg_1)
        cli_bullets(c(
          " " = msg_2,
          " " = run_full
        ))
      }
    }
    cli_end()
  }

  ret
}

find_environments <- function(x) {
  conda_names <- tryCatch(conda_list()$name, error = function(e) character())
  ve_names <- virtualenv_list()
  all_names <- c(ve_names, conda_names)
  sub_names <- substr(all_names, 1, nchar(x))
  matched <- all_names[sub_names == x]
  sorted <- sort(matched, decreasing = TRUE)
  sorted
}
