% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pvlrt-main-function.R
\name{pvlrt}
\alias{pvlrt}
\title{Pseudo Likelihood Ratio Test for determining significant AE-Drug pairs under
Poisson and zero-inflated Poisson models for pharmacovigilance}
\usage{
pvlrt(
  contin_table,
  nsim = 10000,
  omega_vec = NULL,
  zi_prob = NULL,
  no_zi_idx = NULL,
  drug_class_idx = as.list(1:ncol(contin_table)),
  test_drug_idx = 1:ncol(contin_table),
  grouped_omega_est = FALSE,
  test_zi = NULL,
  test_omega = NULL,
  pval_ineq_strict = FALSE,
  parametrization = "rrr",
  null_boot_type = "parametric",
  is_zi_structural = TRUE,
  return_overall_loglik = TRUE,
  ...
)
}
\arguments{
\item{contin_table}{IxJ contingency table showing pairwise counts of adverse
effects for I AE and J Drugs}

\item{nsim}{Number of simulated null contingency table to use for computing
the p-value of the test}

\item{zi_prob, omega_vec}{Alias, determining zero inflation probabilities
in the model. Can be a vector, providing different zero inflation
probabilities for different drugs, or a scalar, providing the common zero
inflation probability for all drugs. If NULL (default), then is estimated from the data. See also the
description of the argument \code{grouped_omega_est}. If \code{omega_vec = rep(0, ncol(contin_table))},
then test reduces to an ordinary (non-zero inflated) Poisson test. NOTE: \code{zi_prob} and \code{omega_vec}
are alias.}

\item{no_zi_idx}{List of pairs {(i, j)} where zero inflation is not allowed. To
specify the entirety i-th row (or j-th column) use c(i, 0) (or c(0, j)). See examples.}

\item{drug_class_idx}{a list, with the h-th entry providing the h-th group/class
of drugs. By default, each drug forms its own class. If more than one drug is
present in a class, an extended LRT is performed. See examples.}

\item{test_drug_idx}{integer vector representing the columns (drug indices) of contin_table to be tested for signal.
Defaults to all columns.}

\item{grouped_omega_est}{Logical. When performing a test with grouped drug classes (extended LRT),
should the estimated zero-inflation parameter "omega" reflect
the corresponding grouping? If TRUE, then the estimated omegas are obtained by combining
columns from the same group, and if FALSE (default), then omegas are estimated separately for each drug (column)
irrespective of the groups specified through  \code{drug_class_idx}. Ignored if \code{omega_vec} is
supplied/non-\code{NULL} (i.e., not estimated).}

\item{test_zi, test_omega}{logical indicators specifying whether to perform a
pseudo likelihood ratio test for zero inflation. Defaults to FALSE. Ignored
if \code{omega_vec} is supplied (is non-NULL). Defaults to FALSE.
NOTE: \code{test_omega} and \code{test_zi} are aliases.}

\item{pval_ineq_strict}{logical. Use a strict inequality in the definition of the p-values?  Defaults to FALSE.}

\item{parametrization}{Type of parametrization to use in the LR test. Available choices are "rrr", "lambda", "rr",
and "p-q". The relative reporting ratio (default) parametrization of the test is used when
when \code{parametrization \%in\% c("rrr", "lambda")}, and the reporting rate parametrization is used
otherwise. NOTE: zero inflation can be handled only for the relative reporting ratio parametrization.}

\item{null_boot_type}{Type of bootstrap sampling to perform for generating null resamples.
Available choices are "parametric" (default) and "non-parametric". NOTE: zero inflation is not
handled properly in a non-parametric bootstrap resampling.}

\item{is_zi_structural}{logical. Do the inflated zeros correspond to structural
zeros (indicating impossible AE-Drug combination)? This determines how the
bootstrap null zero-inflation indicators are generated. If TRUE (default),
then then the null zero-inflation random indicators are randomly generated using the
(estimated) \emph{conditional} probabilities of zero inflation given observed
counts. If FALSE, then they are generated using the \emph{marginal} (drug-specific)
estimated probabilities of zero-inflation.}

\item{return_overall_loglik}{logical. Return overall log-likelihood for the table? This is needed
if \code{logLik} method is to be used.}

\item{...}{additional arguments. Currently unused.}
}
\value{
The function returns an S3 object of class \code{pvlrt} containing  test results. A \code{pvlrt}
object is simply a re-classified matrix containing log likelihood ratio test statistics
for cells in the input contingency table, with various other test and input data information (including
p-values, estimated zero inflation parameters, overall log-likelihood etc.) embedded
as attributes. The following S3 methods and functions are available for an \code{pvlrt} object:

Various postprocessing methods for \code{pvlrt} objects are available. This includes:
\itemize{
\item \link{bubbleplot_pvlrt}
\item \link{extract_AE_names}
\item \link{extract_Drug_names}
\item \link{extract_lrstat_matrix}
\item \link{extract_n_matrix}
\item \link{extract_p_value_matrix}
\item \link{extract_significant_pairs}
\item \link{extract_zi_probability}
\item \link{heatmap_pvlrt}
\item \link{lrt_poisson}
\item \link{lrt_vanilla_poisson}
\item \link{lrt_zi_poisson}
\item \link{r_contin_table_zip}
\item \link{set_AE_names}
\item \link{set_Drug_names}
\item \link{print.pvlrt}
\item \link{plot.pvlrt}
\item \link{summary.pvlrt}
\item \link{logLik.pvlrt}
\item \link{as.matrix.pvlrt}
}
}
\description{
Pseudo Likelihood Ratio Test for determining significant AE-Drug pairs under
Poisson and zero-inflated Poisson models for pharmacovigilance
}
\examples{

data("statin46")

# 500 bootstrap iterations (nsim) in each example below
# are for quick demonstration only --
# we recommended setting nsim to 10000 (default) or bigger

# no grouping -- each drug forms its own class,
# default "rrr" (lambda) parametrization, possible zi,
# null distribution evaluated using parametric bootstrap
test1 <- pvlrt(statin46, nsim = 500)
test1
## extract the observed LRT statistic
extract_lrstat_matrix(test1)
## extract the estimated omegas
extract_zi_probability(test1)

# grouped drugs --
# group 1: drug 1, drug 2
# group 2: drug 3, drug 4
# drug 5, 6, 7 in their own groups
drug_groups <- list(c(1, 2), c(3, 4), 5, 6, 7)
test2 <- pvlrt(statin46, drug_class_idx = drug_groups, nsim = 500)
test2


# specify no zero inflation at the entirety of the last row and the last column
no_zi_idx <- list(c(nrow(statin46), 0), c(0, ncol(statin46)))
test3 <- pvlrt(statin46, no_zi_idx = no_zi_idx, nsim = 500)
test3

\donttest{
# use non-parametric bootstrap to evaluate the null distribution
# takes longer, due to computational costs with non-parametric
# contigency table generation
test4 <- pvlrt(statin46, null_boot_type = "non-parametric", nsim = 500)
test4
}

# test zi probabilities (omegas)
test5 <- pvlrt(statin46, test_omega = TRUE, nsim = 500)
test5


}
