% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pluck.R
\name{pluck}
\alias{pluck}
\alias{pluck<-}
\alias{pluck_exists}
\title{Safely get or set an element deep within a nested data structure}
\usage{
pluck(.x, ..., .default = NULL)

pluck(.x, ...) <- value

pluck_exists(.x, ...)
}
\arguments{
\item{.x, x}{A vector or environment}

\item{...}{A list of accessors for indexing into the object. Can be
an positive integer, a negative integer (to index from the right),
a string (to index into names), or an accessor function
(except for the assignment variants which only support names and
positions). If the object being indexed is an S4 object,
accessing it by name will return the corresponding slot.

\link[rlang:dyn-dots]{Dynamic dots} are supported. In particular, if
your accessors are stored in a list, you can splice that in with
\verb{!!!}.}

\item{.default}{Value to use if target is \code{NULL} or absent.}

\item{value}{A value to replace in \code{.x} at the pluck location.
Use \code{zap()} to instead remove the element.}
}
\description{
\code{pluck()} implements a generalised form of \code{[[} that allow you to index
deeply and flexibly into data structures. (If you're looking for an
equivalent of \code{[}, see \code{\link[=keep_at]{keep_at()}}.) \code{pluck()} always succeeds, returning
\code{.default} if the index you are trying to access does not exist or is \code{NULL}.
(If you're looking for a variant that errors, try \code{\link[=chuck]{chuck()}}.)

\verb{pluck<-()} is the assignment equivalent, allowing you to modify an object
deep within a nested data structure.

\code{pluck_exists()} tells you whether or not an object exists using the
same rules as pluck (i.e. a \code{NULL} element is equivalent to an absent
element).
}
\details{
\itemize{
\item You can pluck or chuck with standard accessors like integer
positions and string names, and also accepts arbitrary accessor
functions, i.e. functions that take an object and return some
internal piece.

This is often more readable than a mix of operators and accessors
because it reads linearly and is free of syntactic
cruft. Compare: \code{accessor(x[[1]])$foo} to \code{pluck(x, 1, accessor, "foo")}.
\item These accessors never partial-match. This is unlike \code{$} which
will select the \code{disp} object if you write \code{mtcars$di}.
}
}
\examples{
# Let's create a list of data structures:
obj1 <- list("a", list(1, elt = "foo"))
obj2 <- list("b", list(2, elt = "bar"))
x <- list(obj1, obj2)

# pluck() provides a way of retrieving objects from such data
# structures using a combination of numeric positions, vector or
# list names, and accessor functions.

# Numeric positions index into the list by position, just like `[[`:
pluck(x, 1)
# same as x[[1]]

# Index from the back
pluck(x, -1)
# same as x[[2]]

pluck(x, 1, 2)
# same as x[[1]][[2]]

# Supply names to index into named vectors:
pluck(x, 1, 2, "elt")
# same as x[[1]][[2]][["elt"]]

# By default, pluck() consistently returns `NULL` when an element
# does not exist:
pluck(x, 10)
try(x[[10]])

# You can also supply a default value for non-existing elements:
pluck(x, 10, .default = NA)

# The map() functions use pluck() by default to retrieve multiple
# values from a list:
map_chr(x, 1)
map_int(x, c(2, 1))

# pluck() also supports accessor functions:
my_element <- function(x) x[[2]]$elt
pluck(x, 1, my_element)
pluck(x, 2, my_element)

# Even for this simple data structure, this is more readable than
# the alternative form because it requires you to read both from
# right-to-left and from left-to-right in different parts of the
# expression:
my_element(x[[1]])

# If you have a list of accessors, you can splice those in with `!!!`:
idx <- list(1, my_element)
pluck(x, !!!idx)
}
\seealso{
\itemize{
\item \code{\link[=attr_getter]{attr_getter()}} for creating attribute getters suitable for use
with \code{pluck()} and \code{chuck()}.
\item \code{\link[=modify_in]{modify_in()}} for applying a function to a plucked location.
\item \code{\link[=keep_at]{keep_at()}} is similar to \code{pluck()}, but retain the structure
of the list instead of converting it into a vector.
}
}
