% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/postgis_insert_update.R
\name{postgis_insert}
\alias{postgis_insert}
\alias{postgis_update}
\title{Insert or update records in a PostgreSQL table from a R data frame.}
\usage{
postgis_insert(conn, df, tbl, write_cols = NA, geom_name = NA_character_,
  hstore_name = NA_character_)

postgis_update(conn, df, tbl, id_cols, update_cols, geom_name = NA_character_,
  hstore_name = NA_character_, hstore_concat = TRUE)
}
\arguments{
\item{conn}{A \code{\link[RPostgreSQL]{PostgreSQLConnection-class}} object,
such as the output of \code{\link[DBI]{dbConnect}}.}

\item{df}{A data frame (if \code{geom_name = NA}) or
Spatial[Points/Lines/Polygons]DataFrame.}

\item{tbl}{Name of the PostgreSQL table to write to.}

\item{write_cols}{A character vector, corresponding to the columns in
\code{df} to insert in the database table. If \code{NA}, inserts all columns.}

\item{geom_name}{Name of the geometry column in the database table
(\code{NA} if none).}

\item{hstore_name}{Name of the hstore column in both \code{df} and the
database table (\code{NA} if none).}

\item{id_cols}{A character vector, corresponding to the columns in \code{df}
used to match records between \code{df} and the database table.}

\item{update_cols}{A character vector, corresponding to the columns that
must be updated in the database table based on values in \code{df}.}

\item{hstore_concat}{If TRUE, hstore columns are updated by concatenation.}
}
\value{
The result of \code{\link[DBI]{dbSendQuery}}.
}
\description{
These functions produce INSERT (\code{postgis_insert}) or UPDATE (\code{postgis_update})
queries to write data from a R data frame to a PostgreSQL table, with options
to include a geometry layer and a list-column of key-value pairs (as a PostgreSQL hstore).
The queries are passed to the database with \code{\link[DBI]{dbSendQuery}}.
}
\details{
All column names used in the query must match between the input data frame and
the target database table (except for \code{geom_name} which only applies to
the table).

\code{postgis_update} creates an \emph{UPDATE ... SET ... FROM ...} query,
which effectively joins the original table and input data frame based on matching
values in \code{id_cols}, then updates the values in \code{update_cols}.
The combination of \code{id_cols} must be unique in \code{df}, but they can
be duplicated in the database table, in which case multiple rows are updated
from a single row in \code{df}. Neither the geometry nor the hstore column can be
used in \code{id_cols}.

Note that if \code{hstore_concat = TRUE} (the default), hstore columns are updated
by \emph{concatenation}, i.e. new keys are added, values associated with
existing keys are updated, no keys are deleted. To overwrite whole hstore
"cells", potentially deleting keys absent in \code{df}, set \code{hstore_concat = FALSE}.
}
\examples{
\dontrun{
library(RPostgreSQL)
con <- dbConnect(PostgreSQL(), dbname = "my_db")

# Returns a SpatialPointsDataFrame
cities <- get_postgis_query(con, "SELECT name, geom, datalist FROM city",
                            geom_name = "geom", hstore_name = "datalist")

# Create a new field in hstore and update DB
cities@data$datalist \%->\% "pop_density" <-
   cities@data$datalist \%->\% "population" / cities@data$datalist \%->\% "area"
postgis_update(con, cities, "city",
               id_cols = "name", update_cols = "datalist",
               geom_name = "geom", hstore_name = "datalist")

# Add rows to DB with postgis_insert
# (new_cities is a SpatialPointsDataFrame with same columns as cities)
postgis_insert(con, new_cities, "city",
               geom_name = "geom", hstore_name = "datalist")
}

}
\seealso{
\code{\link{get_postgis_query}} for the inverse operation
  (read from database to R).
}
